<?php namespace Useresponse\Acl;

use Useresponse\Essentials\PropsFallbackTrait;

/**
 * Class Manager
 *
 * @package Useresponse\Acl
 */
class Manager
{
    use PropsFallbackTrait;

    /**
     * Roles storage.
     *
     * @var Role[]
     */
    protected static $roles = [];

    /**
     * Returns logged user.
     *
     * @return \User
     */
    public static function loggedUser()
    {
        return \Singular_Core::_('Auth')->getLoggedUser();
    }

    /**
     * Returns role by name.
     *
     * @param string $name
     * @return Role
     */
    public static function role($name)
    {
        if (!array_key_exists($name, self::$roles)) {
            self::$roles[$name] = new Role($name);
        }

        return self::$roles[$name];
    }

    /**
     * Returns all permissions.
     *
     * @return Permission[]
     */
    public static function permissions()
    {
        return self::prop(__FUNCTION__, function () {
            $permissions = new \ArrayObject();
            \Singular_Event::dispatch('permissions', $permissions);
            $permissions = $permissions->getArrayCopy();
            $result = [];

            foreach ($permissions as $slug => $permission) {
                if (!empty($permission['depends_on']) && !empty($permissions[$permission['depends_on']])) {
                    $parentSlug = $permission['depends_on'];
                    unset($permission['depends_on']);
                    if (empty($permissions[$parentSlug]['dependencies'])) {
                        $permissions[$parentSlug]['dependencies'] = [];
                    }
                    $permissions[$parentSlug]['dependencies'][$slug] = $permission;
                    unset($permissions[$slug]);
                }
            }

            foreach ($permissions as $slug => $permission) {
                $permission['slug'] = $slug;
                $result[$slug] = new Permission($permission);
            }

            return $result;
        });
    }

    /**
     * Returns permission by name.
     *
     * @param string $name
     * @return Permission
     */
    public static function permission($name)
    {
        $permissions = self::permissions();

        return array_key_exists($name, $permissions) ? $permissions[$name] : new Permission(['slug' => $name]);
    }

    /**
     * Returns roles slugs that granted with specified permission(s).
     *
     * @param array|string $permissions
     * @return array
     */
    public static function rolesSlugsByPermissions($permissions)
    {
        $permissions = is_array($permissions) ? $permissions : [$permissions];
        $roles = [\System_Model_Role::ADMIN, \System_Model_Role::AGENT];

        foreach ($permissions as $permission) {
            $roles = array_merge($roles, self::permission($permission)->rolesSlugs());
        }

        return array_unique($roles);
    }

    public static function allowedEditTime()
    {
        return (int)\Singular_Runtime::extract('config')->resources->acl->allowed_edit_time;
    }
}
