<?php
/**************************************************************
 * Copyright UseResponse Inc., 2012-2016. All Rights Reserved.
 **************************************************************
 * NOTICE:  This source code is the property of UseResponse Inc.
 * In no case shall you rent, lease, lend, redistribute
 * any of below source code to a 3rd party individual or entity.
 *
 * @category   Resources
 * @package    Resources_Model
 * @copyright  UseResponse Inc., 2012-2016 (https://www.useresponse.com)
 * @license    https://www.useresponse.com/eula_license-agreement (Commercial)
 */
use Application\Module\Resources\Models\Transform\CommentInterface;
use Application\Module\Resources\Models\Transform\SingleComment;
use Application\Module\System\Facility\Extra;
use Application\Module\System\Mappers\UserNoteSource;
use Useresponse\Acl\Manager;
use Useresponse\Data\Filter\Emoji;

/**
 * comments row model
 *
 * @uses       System_Model_Row
 * @category   Resources
 * @package    Resources_Model
 *
 * @property int    $id
 * @property string $content
 * @property string $module
 * @property int    $object_id
 * @property int    $user_id
 * @property int    $reply_to
 * @property int    $created_by_id
 * @property string $created_at
 * @property string $updated_at
 * @property int    $updated_by_id
 * @property int    $deleted_by_id
 * @property string $deleted_at
 * @property int    $state
 * @property int    $original_state
 * @property int    $abuse_or_spam
 * @property int    $source
 * @property int    $is_private
 * @property string $reminder
 * @property string $remind_at
 */
class Resources_Model_Comment extends System_Model_Row
    implements System_Lib_Interface_Owner, System_Lib_Interface_Attachments
{
    use System_Lib_EmailContentTrait;

    CONST owner = 'comment';

    /**
     * @var string
     */
    public $type = 'comment';

    /**
     * Table class
     *
     * @var string
     */
    protected $_tableClass = 'Resources_Model_DbTable_Comments';

    /**
     * @var int
     */
    protected $_votesSum = null;

    protected $_replies = null;

    /**
     * @var System_Model_User
     */
    private $_author = false;

    private $saveSilent = false;

    /**
     * Allows pre-insert or pre-update logic to be applied to row.
     * Subclasses may override this method.
     *
     * @return void
     * @throws \OutOfRangeException
     * @throws \InvalidArgumentException
     */
    protected function write()
    {
        parent::write();
        $this->content = (new Emoji)->filter($this->content);
    }

    /**
     * Allows post-insert logic to be applied to row.
     * Subclasses may override this method.
     *
     * @see    parent::_postInsert()
     * @return void
     * @throws \Zend_Db_Table_Row_Exception
     */
    public function _postInsert()
    {
        if ($this->user_id > 0) {
            parent::_postInsert();
            return;
        }

        if ($this->module !== 'helpdesk') {
            $vote            = new Resources_Model_Vote();
            $vote->owner     = $this->getOwner();
            $vote->object_id = $this->id;
            $vote->value     = 1;
            $vote->save();
        }

        if ($this->saveSilent) {
            parent::_postInsert();
            return;
        }

        $parent = $this->getParent();

        // Log
        $author = Manager::loggedUser();
        $log    = new System_Model_Log();

        if (Singular_Core::_('Auth')->getLoggedUser()->isGuest() || Singular_Core::_('Environment')->isScheduleTask()) {
            $author             = Singular_Loader::get('Users')->findById($this->created_by_id);
            $log->created_by_id = $this->created_by_id;
        }

        $log->save(
            [
                'owner'        => $this->getOwner(),
                'object_id'    => $this->id,
                'action'       => (int)$this->is_private ? System_Model_DbTable_Logs::ADD_NOTE : System_Model_DbTable_Logs::COMMENTED,
                'object_title' => $parent->title,
                'object_type'  => $parent->object_type,
                'is_private'   => (int)($parent->is_private || $this->is_private),
            ]
        );

        if ($author instanceof User) {
            $author->logActivity();
        }

        parent::_postInsert();
        if ((int)$this->is_private === 0) {
            if (empty($parent->replied_at) && $this->getAuthor()->role()->isAgent()) {
                $parent->replied_at    = date(Singular_Date::FORMAT_MYSQL_DATETIME);
                $parent->replied_by_id = $this->getAuthor()->id;
                $parent->save();
            }
            $parent->rebuildCommentedField();
        }

        Singular_Loader::get('Notifications')->markAsReadByObject(self::owner, $this->id, $this->created_by_id);

        Singular_Core::_('Cache')->getCache()->remove($parent->getCacheId('comments_0'));
        Singular_Core::_('Cache')->getCache()->remove($parent->getCacheId('comments_1'));
        $this->_cleanDefaultCache();
    }

    /**
     * Returns owner name
     *
     * @return mixed
     */
    public function getOwner()
    {
        return self::owner;
    }

    /**
     * Returns object author
     *
     * @return System_Model_User
     */
    public function getAuthor()
    {
        if (!$this->_author) {
            $this->_author = Singular_Loader::get('Users')->findById($this->created_by_id);
        }

        return $this->_author;
    }

    /**
     * Returns object by object_id
     *
     * @deprecated
     * @return System_Model_Object
     */
    public function getObject()
    {
        if (!isset($this->_properties['Object'])) {
            $result = Singular_Loader::get('Objects')->find($this->object_id)->current();
            if ($result instanceof System_Model_Row) {
                $this->_properties['Object'] = $result;
            }

        }

        return $this->_properties['Object'];
    }

    /**
     * Checks if current comment is negative
     *
     * @return int
     */
    public function isNegative()
    {
        return $this->is_negative;
    }

    /**
     * Returns total number of votes for current comment
     *
     * @return int
     */
    public function getVotesSum()
    {
        if (is_null($this->_votesSum)) {
            $votes           = Singular_Loader::get('Resources_Model_DbTable_Votes');
            $this->_votesSum = $votes->getRatingByObject($this);
        }

        return $this->_votesSum;
    }

    /**
     * Checks if given user (field:user_id)
     * is already voted for current comment
     *
     * @param  int|null $userId
     * @return bool
     */
    public function isVoted($userId = null)
    {
        if (!$userId) {
            $auth = Singular_Core::_("Auth")->getLoggedUser();
            if (!$auth->isGuest()) {
                $userId = $auth->id;
            } elseif (Singular_Core::_("Settings")->get("anonymous_votes.value")) {
                $cookie = Zend_Controller_Front::getInstance()->getRequest()
                    ->getCookie('votes', '{}');
                $cookie = Zend_Json_Decoder::decode($cookie);

                return isset($cookie['comment'][$this->id]) ? true : false;
            } else {
                return true;
            }
        }
        $validator = new Zend_Validate_Db_RecordExists(
            [
                'table' => Singular_Loader::get('Votes')->getTableName(),
                'field' => 'created_by_id'
            ]);
        $validator->getSelect()->where('owner=?', $this->getowner())
            ->where('object_id=?', $this->id);

//        $votes = Singular_Loader::get('Resources_Model_DbTable_Votes');
//        $vote = $votes->findByObjectAndUserId($this, $userId);

        return $validator->isValid($userId);
    }

    /**
     * Checks if logged user can edit comments
     *
     * @return bool
     */
    public function canEdit()
    {
        $loggedUser = Manager::loggedUser();

        if ((int)$this->is_private > 0) {
            $result = $loggedUser->role()->isAgent() && (int)$loggedUser->id === (int)$this->created_by_id || $loggedUser->role()->isAdmin();
        } else {
            $result = $loggedUser->role()->isAgent();

            if (!$result) {
                $loggedUser = Singular_Core::_('Auth')->getLoggedUser();

                if ($loggedUser->id == $this->created_by_id) {
                    $createdAt = strtotime($this->created_at);
                    $editTime  = Manager::allowedEditTime();

                    if ($editTime < 0 || $createdAt > time() - $editTime) {
                        $result = true;
                    }
                }
            }
        }

        $data          = new stdClass();
        $data->comment = $this;
        $data->result  = $result;
        Singular_Event::dispatch('canEditComment', $data);

        return $data->result;
    }

    /**
     * Checks if logged user can delete comments
     *
     * @param System_Model_User $user
     * @return bool
     */
    public function canDelete(System_Model_User $user)
    {
        return (int)$this->is_private > 0
            ? $user->role()->isAgent() && (int)$user->id === (int)$this->created_by_id || $user->role()->isAdmin()
            : $user->role()->isAgent();
    }

    /**
     * Returns true if user can see comments
     *
     * @param System_Model_User $user
     * @return boolean
     */
    public function canSee(System_Model_User $user)
    {
        return $this->getParent()->canSee($user);
    }

    /**
     * Returns comment url
     *
     * @return string
     */
    public function getUrl()
    {
        return ($this->getParent() ? $this->getParent()->getUrl() . "#comment-" . $this->id : '#');
    }

    /**
     * Returns comment url for agent
     *
     * @return string
     */
    public function getAgentUrl()
    {
        if ($this->user_id > 0) {
            return view()->url(['user_id' => $this->user_id], 'agent_user_profile') . '#notes';
        }

        return ($this->getParent() ? $this->getParent()->getAgentUrl() . "#comment-" . $this->id : '#');
    }

    /**
     * Return attachments of the object
     *
     * @return System_Model_Rowset|File[]
     */
    public function getAttachments()
    {
        $select      = $this->select()->where('owner = ?', static::getOwner());
        $attachments = $this->findDependentRowset('Resources_Model_DbTable_Files', static::getOwner(), $select);

        return count($attachments) ? $attachments : null;
    }

    /**
     * Return images of the object
     *
     * @return System_Model_Rowset
     */
    public function getImages()
    {
        $select = $this->select()->where('owner = ?', $this->getOwner() . '_img');
        $images = $this->findDependentRowset('Resources_Model_DbTable_Files', static::getOwner(), $select);

        return count($images) ? $images : null;
    }

    /**
     * Deletes existing rows.
     *
     * @return int The number of rows deleted.
     */
    public function deleteWithRelations()
    {
        $commentsDb = Singular_Loader::get('Comments');
        $select     = $commentsDb->select();
        $select->where("id = {$this->id} OR reply_to = {$this->id}");
        $comments = $commentsDb->fetchAll($select);
        if (count($comments) > 0) {
            foreach ($comments as $comment) {
                /** @var $comment Resources_Model_Comment */
                $condition        = "`owner` = ? AND `object_id` = ?";
                $bind             = [$comment->getOwner(), $comment->id];
                $attachmentsTable = Singular_Core::_('Db')->getTablePrefix('attachments');
                $files            = Singular_Core::_('Db')->getAdapter()->fetchCol("SELECT `location` FROM `{$attachmentsTable}` WHERE {$condition}", $bind);
                foreach ($files as $file) {
                    @unlink(PUBLIC_PATH . "/attachments/{$file}");
                }
                Singular_Core::_('Db')->getAdapter()->query("DELETE FROM `{$attachmentsTable}` WHERE {$condition}", $bind);
                Singular_Loader::get('Votes')->delete("owner = '{$comment->getOwner()}' AND object_id = '{$comment->id}'");
                $comment->delete();
            }
        }
    }

    /**
     * Move comment and it's replies to Trash
     *
     * @return void
     */
    public function moveToTrash()
    {
        $commentsDb = Singular_Loader::get('Comments');
        $select     = $commentsDb->getSelect();
        $select->where('reply_to = ' . (int)$this->id);
        $comments = $commentsDb->fetchAll($select);
        System_Mapper_State::object($this)->moveToTrash();
        if (count($comments) > 0) {
            foreach ($comments as $comment) {
                /** @var Resources_Model_Comment */
                System_Mapper_State::object($comment)->moveToTrash();
            }
        }

        if ($this->object_id > 0) {
            $this->getParent()->rebuildCommentedField();
            Singular_Core::_('Cache')->getCache()->remove($this->getParent()->getCacheId('comments_0'));
            Singular_Core::_('Cache')->getCache()->remove($this->getParent()->getCacheId('comments_1'));
        }
    }

    /**
     * Restore comment and it's replies from Trash
     *
     * @return void
     */
    public function restoreFromTrash()
    {
        $commentsDb = Singular_Loader::get('Comments');
        $select     = $commentsDb->select();
        $select->where('reply_to = ' . (int)$this->id);
        $select->where('`state` = ' . System_Mapper_State::TRASHED);
        $comments = $commentsDb->fetchAll($select);
        System_Mapper_State::object($this)->makeNormal();
        if (count($comments) > 0) {
            foreach ($comments as $comment) {
                /** @var Resources_Model_Comment */
                System_Mapper_State::object($comment)->makeNormal();
            }
        }
        $this->getParent()->rebuildCommentedField();
        Singular_Core::_('Cache')->getCache()->remove($this->getParent()->getCacheId('comments_0'));
        Singular_Core::_('Cache')->getCache()->remove($this->getParent()->getCacheId('comments_1'));
    }

    /**
     * Converts current object's data to third-party applications using notation.
     *
     * @param CommentInterface $dataSource
     * @return array
     */
    public function transform(CommentInterface $dataSource = null)
    {
        /** @var Singular_View $view */
        $view = Zend_Layout::getMvcInstance()->getView();
        $view->additionalModulePath('system');

        if ($dataSource === null) {
            $dataSource = new SingleComment(loggedUser());
        }

        if ($this->user_id > 0) {
            return [
                'id'         => (int)$this->id,
                'content'    => $view->bbcode($this->content, true, $this),
                'userId'     => (int)$this->object_id,
                'author'     => $dataSource->author($this),
                'createdAt'  => $this->getCreatedAt(true),
                'remindAt'   => $this->getRemindAt(true),
                'createdAgo' => $view->t(
                    ':relativeDate ago',
                    ['relativeDate' => $view->relativeDateTime($this->created_at, true)]
                ),
                'source'     => $this->source > 0 ? UserNoteSource::getList()[(int)$this->source] : null
            ];
        }

        $data = [
            'id'         => (int)$this->id,
            'ownership'  => $this->module,
            'content'    => $view->bbcode($this->content, true, $this),
            'objectId'   => (int)$this->object_id,
            'replyTo'    => $dataSource->replies($this),
            'replyToId'  => (int)$this->reply_to,
            'author'     => $dataSource->author($this),
            'createdAt'  => $this->getCreatedAt(true),
            'updatedAt'  => $this->getUpdatedAt(true),
            'createdAgo' => $view->t(
                ':relativeDate ago',
                ['relativeDate' => $view->relativeDateTime($this->created_at, true)]
            ),
            'is_best'    => (int)$dataSource->isBest($this)
        ];

        $votes = $dataSource->votes($this);
        $loggedUser = loggedUser();

        $data['votes'] = [
            'amount'  => $votes,
            'plural'  => $votes === 1 ? $view->t('vote') : $view->t('votes'),
            'isVoted' => $dataSource->isVoted($this),
            'canVote' => $this->module !== 'helpdesk' && !$loggedUser->isGuest() && $loggedUser->id != $this->created_by_id
        ];

        return $data;
    }

    /**
     * Returns replies it's comment
     *
     * @return Resources_Model_Comment[]
     */
    public function getReplies()
    {
        if (!$this->_replies) {
            if ((bool)$this->reply_to) {
                return null;
            }
            $this->_replies = $this->getTable()->fetchAll(
                $this->getTable()->select()->where("reply_to=?", $this->id)
            );
        }

        return $this->_replies;
    }

    /**
     * Returns ISO-formatted created_at date
     *
     * @param  bool $localeFormatted
     * @return Zend_Date|string
     */
    public function getCreatedAt($localeFormatted = false)
    {
        if ($localeFormatted) {
            return Zend_Layout::getMvcInstance()->getView()->dateFormat(
                $this->created_at
            )->fromMysql();
        }

        return new Zend_Date($this->created_at, Zend_Date::ISO_8601);
    }

    /**
     * Returns ISO-formatted updated_at date
     *
     * @param  bool $localeFormatted
     * @return Zend_Date
     */
    public function getUpdatedAt($localeFormatted = false)
    {
        if ($localeFormatted) {
            return Zend_Layout::getMvcInstance()->getView()->dateFormat(
                $this->updated_at
            )->fromMysql();
        }

        return new Zend_Date($this->updated_at, Zend_Date::ISO_8601);
    }

    /**
     * Returns ISO-formatted created_at date
     *
     * @param  bool $localeFormatted
     * @return Zend_Date|string
     */
    public function getRemindAt($localeFormatted = false)
    {
        if ($localeFormatted) {
            return Zend_Layout::getMvcInstance()->getView()->dateFormat(
                $this->remind_at
            )->fromMysql();
        }

        return new Zend_Date($this->remind_at, Zend_Date::ISO_8601);
    }

    /**
     * Save comment
     *
     * @param null|array $data
     * @return mixed
     * @throws Exception
     */
    public function save($data = null)
    {
        $loggedUser = Singular_Core::_('Auth')->getLoggedUser();

        if ($loggedUser->role()->isAgent() && is_array($data) && !empty($data['is_private']) && !$this->isNew()) {
            Singular_Core::_('Db')->getAdapter()->update($this->getTable()->getTableName(), ['reply_to' => 0], "reply_to = '{$this->id}'");
        }

        if ($loggedUser->isGuest() && $loggedUser->getAccessKey()) {
            $sharing = Singular_Loader::get('Sharing')->findByAccessKey($loggedUser->getAccessKey());

            if (!($sharing instanceof Resources_Model_Share)) {
                Singular_Core::_('Environment')->accessDenied();
            }

            $user = Singular_Loader::get('Users')->findByEmail($sharing->email);

            if (!($user instanceof System_Model_User)) {
                $user = new System_Model_User();
                list($emailName,) = explode('@', $sharing->email);
                $userData = [
                    'full_name' => $emailName,
                    'email'     => $sharing->email,
                    'password'  => Singular_Core::_('Auth')->passwordHash(time() . rand(10, 99))
                ];
                $user->registerAnonymous($userData);

                if (is_array($data)) {
                    $this->setFromArray($data);
                }
            }

            $user->save(['logged_at' => date('Y-m-d H:i:s', time())]);
            $this->created_by_id = $user->id;
        } elseif (
            !empty($data['module']) &&
            $data['module'] === 'feedback' &&
            loggedUser()->role()->isGuest() &&
            Extra::feedbackAnonymousCommentsIsAllowed() &&
            Singular_Core::_('Settings')->isOn('anonymous_topic_comments')
        ) {
            $activateAnonymous = Singular_Core::_('Settings')->isOn('anonymous_auto_activate');

            if (!empty($data['notify_email'])) {
                $user = Singular_Loader::get('System_Model_DbTable_Users')->findByEmail($data['notify_email']);

                if (!($user instanceof System_Model_User)) {
                    $user = new System_Model_User();

                    if (!empty($data['full_name'])) {
                        $fullName = $data['full_name'];
                    } else {
                        $emailParts = explode('@', $data['notify_email']);
                        $fullName   = $emailParts[0];
                    }

                    $userData = [
                        'full_name' => $fullName,
                        'email'     => $data['notify_email'],
                    ];

                    $registered = $activateAnonymous
                        ? $user->register($userData)
                        : $user->registerAnonymous($userData);

                    if (!$registered) {
                        return false;
                    }
                } elseif ($activateAnonymous && $user->isAnonymous()) {
                    $newPass = System_Service_Auth::randPassword();
                    $user->setCleanPassword($newPass);
                    $user->setPassword($newPass);
                    $user->state = System_Model_User::STATE_NORMAL;
                    $user->setApiKey(System_Service_Api::generate());
                    $user->save();
                    Singular_Event::dispatch('afterRegistration', $user);
                }

                if ((int)$user->state === System_Model_User::STATE_BANNED) {
                    throw new Exception('This account is banned!');
                }

                $data['created_by_id'] = $user->id;
            } else {
                $anonymousUser         = Singular_Loader::get('System_Model_DbTable_Users')->findAnonymous();
                $data['created_by_id'] = $anonymousUser->id;
            }
        }

        if (is_array($data)) {
            $this->setFromArray($data);
        }

        if (
            !$loggedUser->isGuest() &&
            !$loggedUser->role()->isAgent() &&
            !Singular_Loader::get('Sharing')->exists($loggedUser->email, $this->object_id) &&
            $this->module === 'helpdesk'
        ) {
            (new Resources_Model_Share())->save([
                'module'     => $this->module,
                'email'      => $loggedUser->email,
                'object_id'  => $this->object_id,
                'access_key' => Singular_Loader::get('Sharing')->generateKey(),
            ]);
        }

        if (empty($data['created_by_id']) && empty($this->created_by_id) && $this->isNew()) {
            $data['created_by_id'] = $loggedUser->id;
        }

        return parent::save($data);
    }

    /**
     * Can user download attachment.
     *
     * @param System_Model_User $user
     * @return bool
     */
    public function canDownloadAttachment(System_Model_User $user)
    {
        $parent = $this->getParent();

        return empty($parent) || ($parent instanceof System_Model_Object && $parent->canDownloadAttachment($user));
    }

    /**
     * Can user delete attachment.
     *
     * @param System_Model_User $user
     * @return bool
     */
    public function canDeleteAttachment(System_Model_User $user)
    {
        return
            $this->created_by_id == $user->id ||
            $user->role()->isAgent() ||
            ($this->getParent() instanceof System_Model_Object && $this->getParent()->canDeleteAttachment($user));
    }

    public function saveSilent($silent)
    {
        $this->saveSilent = (bool)$silent;
    }

    /**
     * Allows pre-insert logic to be applied to row.
     * Subclasses may override this method.
     *
     * @see    parent::_insert()
     * @return void
     */
    protected function _insert()
    {
        if (is_null($this->source)) {
            $this->source = System_Mapper_Source::determineSource();
        }

        parent::_insert();
    }

    /**
     * Allows post-delete logic to be applied to row.
     * Subclasses may override this method.
     *
     * @see    parent::_postDelete()
     * @return void
     */
    protected function _postDelete()
    {
        parent::_postDelete();

        // Remove relative votes
        $parent = $this->getParent();

        if ($parent instanceof System_Model_Row) {
            Singular_Core::_('Cache')->getCache()->remove($this->getParent()->getCacheId('comments_0'));
            Singular_Core::_('Cache')->getCache()->remove($this->getParent()->getCacheId('comments_1'));
        }

        $votesDb = Singular_Loader::get("Votes");
        $owner   = self::owner;
        $votes   = $votesDb->fetchAll("object_id={$this->id} AND owner='$owner'");
        if ($votes->valid()) {
            foreach ($votes as $vote) {
                $vote->delete();
            }
        }

        Singular_Loader::get('Notifications')->deleteByObjectId($this->id, $owner);

        Singular_Loader::get('SocialAssigns')->orphanize(System_Model_DbTable_SocialAssigns::TYPE_COMMENT, $this->id);

        $this->_cleanDefaultCache();
    }

    /**
     * Returns object by object_id
     *
     * @return System_Model_Object
     */
    public function getParent()
    {
        if (!isset($this->_properties['parent'])) {
            $this->_properties['parent'] = null;
            $modules                     = Singular_Runtime::extract('active_modules');
            $result                      = null;
            if ((bool)array_search($this->module, $modules)) {

                $model = System_Mapper_Object::getObjectName($this->module);
                if (is_string($model)) {
                    $model = Singular_Loader::get($model);
                }
                $result = $model->find($this->object_id)->current();
            }
            if ($result instanceof System_Model_Row) {
                if ($result instanceof System_Model_Object) {
                    $result = $result->convertObject();
                }
                $this->_properties['parent'] = $result;
            }
        }

        return $this->_properties['parent'];
    }

    /**
     * Clean default cache
     *
     * @return void
     */
    protected function _cleanDefaultCache()
    {
        $activeObject = $this->getParent();
        $cacheTags    = ['navigation', 'comment_' . $this->id, 'comments', 'homepage'];

        if ($activeObject instanceof System_Model_Object) {
            $cacheTags[] = $activeObject->getOwner() . '_' . $activeObject->id;
        }

        Singular_Core::_('Cache')->getCache()
            ->clean(Zend_Cache::CLEANING_MODE_MATCHING_ANY_TAG, $cacheTags);

        if (class_exists('Supercache', false)) {
            Singular_Core::_('Cache')->getCache()->clean(Zend_Cache::CLEANING_MODE_ALL);
        }
    }

    /**
     * Allows post-update logic to be applied to row.
     * Subclasses may override this method.
     *
     * @return void
     */
    protected function _postUpdate()
    {
        parent::_postUpdate();

        if ($this->object_id > 0) {
            Singular_Core::_('Cache')->getCache()->remove($this->getParent()->getCacheId('comments_0'));
            Singular_Core::_('Cache')->getCache()->remove($this->getParent()->getCacheId('comments_1'));
        }

        $this->_cleanDefaultCache();
    }
}
