<?php
/**************************************************************
 * Copyright UseResponse Inc., 2012-2016. All Rights Reserved.
 **************************************************************
 * NOTICE:  This source code is the property of UseResponse Inc.
 * In no case shall you rent, lease, lend, redistribute
 * any of below source code to a 3rd party individual or entity.
 *
 * @category   Feedback
 * @package    Feedback_Model
 * @copyright  UseResponse Inc., 2012-2016 (https://www.useresponse.com)
 * @license    https://www.useresponse.com/eula_license-agreement (Commercial)
 */

use Application\Module\System\Facility\Extra;
use Application\Module\System\Models\Transform\ObjectInterface;
use Application\Module\System\Models\Transform\SingleObject;
use Useresponse\Acl\Manager as AclManager;
use Useresponse\Application\App;
use Useresponse\Application\Http;
use Useresponse\Features\Features;

/**
 * topics (objects) row class
 *
 * @uses       System_Model_Object
 * @category   Feedback
 * @package    Feedback_Model
 */
class Feedback_Model_Topic extends System_Model_Object implements
    System_Lib_Interface_Owner,
    System_Lib_Interface_ObjectsConventions
{
    const MODULE      = 'feedback';
    const OBJECT_NAME = 'topic';

    /* Filter constants */
    const FILTER_CREATED     = 'created';
    const FILTER_RESPONSIBLE = 'responsible';
    const FILTER_FOLLOWING   = 'following';
    const FILTER_COMMENTED   = 'commented';
    const FILTER_VOTED       = 'voted';

    /* Object type constants */
    const OBJECT_TYPE_IDEA     = 'idea';
    const OBJECT_TYPE_PROBLEM  = 'problem';
    const OBJECT_TYPE_QUESTION = 'question';
    const OBJECT_TYPE_THANKS   = 'thanks';

    protected $_module = self::MODULE;

    /**
     * Name of the class of the Zend_Db_Table_Abstract object.
     *
     * @var string
     */
    protected $_tableClass = 'Feedback_Model_DbTable_Topics';

    /**
     * @var Resources_Model_ObjectType
     */

    protected $_type = false;
    protected $_hasVotes = null;

    /**
     * @var Resources_Model_Comment
     */
    private $_lastReply = false;

    /**
     * Return true if topic can have negative votes
     *
     * @return boolean
     */
    public function hasNegativeVote()
    {
        switch ($this->object_type) {
            case self::OBJECT_TYPE_IDEA:
            case self::OBJECT_TYPE_PROBLEM:
                return true;
            default:
                return false;
        }
    }

    /**
     * Return phrases for button's vote
     *
     * @return string
     */
    public function textButtonVote()
    {
        switch ($this->object_type) {
            case self::OBJECT_TYPE_IDEA:
                return 'I like this idea';
            case self::OBJECT_TYPE_PROBLEM:
                return 'The same problem';
            case self::OBJECT_TYPE_QUESTION:
                return 'The same question';
            case self::OBJECT_TYPE_THANKS:
                return 'Thank you too';
        }
    }

    /**
     * Allows pre-insert logic to be applied to row.
     * Subclasses may override this method.
     *
     * @see    parent::_insert()
     * @return void
     */
    protected function _insert()
    {
        $this->score = Feedback_Service_Trending::VOTE_BONUS;

        // dbg triggers
        /*if (
            Singular_Loader::get('Forums')->getActive()->isModerated() &&
            !$this->getAuthor()->role()->isAgent() &&
            !(bool)$this->getAuthor()->is_trusted &&
            $this->module !== 'helpdesk'
        ) {
            \Useresponse\Application\App::services()->get('triggers')->trigger('topic_on_moderation', [
                'object' => $this
            ]);
        }*/

        parent::_insert();
    }

    /**
     * Deletes existing rows.
     *
     * @return int The number of rows deleted.
     */
    function delete()
    {
        $select = $this->select()->where('owner = ? OR owner = ?', $this->getOwner() . '_img', $this->getOwner());
        $files  = $this->getParent()->findDependentRowset('Resources_Model_DbTable_Files', 'Object', $select);

        if (count($files)) {
            foreach ($files as $file) {
                $file->delete();
            }
        }

        return parent::delete();
    }

    /**
     * Returns url.
     *
     * @var string
     * @return string
     */
    public function getUrl()
    {
        if (!$this->_url) {
            $router = Zend_Controller_Front::getInstance()->getRouter();
            try {
                $this->_url = $router->assemble(
                    array(
                        'slug' => $this->slug
                    ),
                    'topic_show'
                );
            } catch (Zend_Exception $e) {
                return '#';
            }
        }

        if (Features::isOn(Features::FEATURE_FORUMS) && !empty($this->forum_id)) {
            $this->_url = Forums_Service_Forums::modifyBaseUrl($this->_url, $this->forum_id);
        }

        return $this->_url;
    }

    public function getEditUrl()
    {
        $router = Zend_Controller_Front::getInstance()->getRouter();

        return $router->assemble(
            array(
                'slug' => $this->slug
            ),
            'topic_edit'
        );
    }

    /**
     * Return the last topic reply.
     *
     * @return Resources_Model_Comment
     */
    public function getLastReply()
    {
        if (!$this->_lastReply) {
            $comment = $this->getLastComment();
            if ($comment instanceof Resources_Model_Comment) {
                $this->_lastReply = $comment;
            } else {
                $this->_lastReply = $this;
            }
        }

        return $this->_lastReply;
    }

    /**
     * Returns true if user can edit topic
     *
     * @param System_Model_User $user
     *
     * @return boolean
     */
    public function canEdit(System_Model_User $user)
    {
        $editTime = AclManager::allowedEditTime();

        return self::canManage($user) ||
            (
                (int)$user->id === (int)$this->created_by_id &&
                ($editTime < 0 || strtotime($this->created_at) > time() - $editTime)
            );
    }

    /**
     * Returns true if user can manage topic
     *
     * @param System_Model_User $user
     *
     * @return boolean
     */
    public function canManage(System_Model_User $user)
    {
        return $user->role()->isAgent();
    }

    /**
     * Returns true if user can delete topic
     *
     * @deprecated
     *
     * @param System_Model_User $user
     *
     * @return boolean
     */
    public function canDelete(System_Model_User $user)
    {
        return self::canManage($user);
    }

    /**
     * Returns true if user can change status of topic
     *
     * @deprecated
     *
     * @param System_Model_User $user
     *
     * @return boolean
     */
    public function canChangeStatus(System_Model_User $user)
    {
        return self::canManage($user);
    }

    /**
     *
     * @deprecated
     *
     * @param System_Model_User $user
     *
     * @return boolean
     */
    public function canLockComments(System_Model_User $user)
    {
        return self::canManage($user);
    }

    /**
     * Returns users
     *
     * @return Zend_Db_Table_Rowset|null
     */
    protected function _findUsersCommentAdd()
    {
        $subs = Singular_Loader::get('Subscribers')->findUsersByObject($this);

        $notifyUsers = array();
        foreach ($subs as $user) {
            if ($this->canSee($user, false)) {
                $notifyUsers[] = $user;
            }
        }

        return $notifyUsers;
    }

    /**
     * Returns topic's type.
     *
     * @return Resources_Model_ObjectType
     */
    public function getType()
    {
        if (!$this->_type) {
            $this->_type = Singular_Loader::get("ObjectTypes")
                ->findActive(Feedback_Model_Topic::MODULE)->findItemsByField("slug", $this->object_type)->current();
        }

        return $this->_type;
    }

    /**
     * Returns url on type of topic
     *
     * @return string
     */
    public function getTypeImage()
    {
        if ($this->getType()) {
            return $this->getType()->getImage();
        }
        return '#';
    }

    /**
     * Is Topic type has votes?
     *
     * @return bool
     */
    public function hasVotesType()
    {
        if (!$this->_hasVotes) {
            $this->_hasVotes = Singular_Loader::get("ObjectTypes")->hasVotesByType($this->object_type);
        }
        return $this->_hasVotes;
    }

    /**
     * Allows post-delete logic to be applied to row.
     * Subclasses may override this method.
     *
     * @see    parent::_postDelete()
     * @return void
     */
    protected function _postDelete()
    {
        parent::_postDelete();

        // Remove relative votes
        $votes = Singular_Loader::get("Votes")->fetchAll("object_id={$this->id} AND owner='{$this->getOwner()}'");
        if ($votes->valid()) {
            foreach ($votes as $vote) {
                $vote->delete();
            }
        }
    }

    /**
     * Allows pre-update logic to be applied to row.
     * Subclasses may override this method.
     *
     * @event  update
     * @return void
     */
    protected function _update()
    {
        parent::_update();
        $this->_status = null;
        $status = $this->getStatus();

        if ($status && $status->object_type != $this->object_type) {
            $defStatus = $this->getDefaultStatus();

            if ($defStatus instanceof System_Model_Status) {
                $this->status_id = $defStatus->id;
            }
        }
    }

    public function votersMap()
    {
        $map = [
            'positive' => [],
            'negative' => []
        ];

        $votes = $this->getVotes();

        foreach ($votes as $vote) {
            if ($vote->value > 0) {
                $section = 'positive';
            } elseif ($vote->value < 0) {
                $section = 'negative';
            } else {
                continue;
            }

            $userId = (int)$vote->created_by_id;

            if ($userId === 0) {
                $userId = User::ANONYMOUS_ID;
            }

            if (array_key_exists($userId, $map[$section])) {
                $map[$section][$userId]->votes += abs((int)$vote->value);
            } else {
                $user = Singular_Loader::get('Users')->findById($userId);
                if ($user instanceof User) {
                    $map[$section][$userId] = (object)[
                        'user'  => $user,
                        'votes' => abs((int)$vote->value)
                    ];
                }
            }
        }

        return $map;
    }

    /**
     * Return restore from archive url.
     *
     * @return string
     */
    public function getRestoreFromArchiveUrl()
    {
        if (isset($this->slug)) {
            $view = Zend_Layout::startMvc()->getView();
            return $view->url(
                array("object_id" => $this->id), 'object_recovery'
            );
        }

        return '#';
    }

    /**
     * Returns restore from trash url.
     *
     * @return string
     */
    public function getRestoreFromTrashUrl()
    {
        if (isset($this->slug)) {
            $view = Zend_Layout::startMvc()->getView();

            return $view->url(
                array("object_id" => $this->id), 'object_recovery'
            );
        }

        return '#';
    }

    /**
     * Saves the properties to the database.
     *
     * @param  array|null $data
     *
     * @return bool|mixed
     */
    public function save($data = null)
    {
        $loggedUser = Singular_Core::_("Auth")->getLoggedUser();

        $moveCommentToTopic = !empty($data['comment_to_object']) && $loggedUser->role()->isAgent()
            ? Singular_Loader::get('Comments')->findById($data['comment_to_object'])
            : null;

        if (!empty($data) && is_array($data) && $this->isNew()) {
            $anonymousTopics = Singular_Core::_('Settings')->isOn('anonymous_topics');
            $activateAnonymous = Singular_Core::_('Settings')->isOn('anonymous_auto_activate');

            if (
                $loggedUser->isGuest() && $anonymousTopics ||
                (Http::isSdkCall() || $loggedUser->role()->isAgent()) && !empty($data['notify_email'])
            ) {
                /* add user */
                if (!empty($data['notify_email'])) {
                    $user = Singular_Loader::get('System_Model_DbTable_Users')->findByEmail($data['notify_email']);

                    if (!($user instanceof System_Model_User)) {
                        $user = new System_Model_User();

                        if (!empty($data['full_name'])) {
                            $fullName = $data['full_name'];
                        } else {
                            $emailParts = explode('@', $data['notify_email']);
                            $fullName = $emailParts[0];
                        }

                        $userData  = array(
                            'full_name' => $fullName,
                            'email'     => $data['notify_email'],
                        );

                        $registered = $activateAnonymous && $anonymousTopics
                            ? $user->register($userData)
                            : $user->registerAnonymous($userData);

                        if (!$registered) {
                            return false;
                        }
                    } elseif ($user->isAnonymous() && $anonymousTopics && $activateAnonymous) {
                        $newPass = System_Service_Auth::randPassword();
                        $user->setCleanPassword($newPass);
                        $user->setPassword($newPass);
                        $user->state = System_Model_User::STATE_NORMAL;
                        $user->setApiKey(System_Service_Api::generate());
                        $user->save();
                        Singular_Event::dispatch('afterRegistration', $user);
                    }

                    if ((int)$user->state === System_Model_User::STATE_BANNED) {
                        throw new Exception('This account is banned!');
                    }

                    $data['created_by_id'] = $user->id;
                } else {
                    $anonymousUser         = Singular_Loader::get('System_Model_DbTable_Users')
                        ->findAnonymous();
                    $data['created_by_id'] = $anonymousUser->id;
                }
            }
        }

        $objectType = is_array($data) && array_key_exists('object_type', $data) ? $data['object_type'] : $this->object_type;
        $statusId = is_array($data) && array_key_exists('status_id', $data) ? $data['status_id'] : $this->status_id;
        $status = $statusId ? Singular_Loader::get('Statuses')->findById($statusId)
            : Singular_Loader::get('Statuses')->findDefaultByObjectType($objectType, $this->module);

        $data['status_id'] = $status instanceof System_Model_Status && $status->object_type === $objectType ? $status->id : 0;

        if (!($status instanceof System_Model_Status) || $status->is_closed) {
            if (!$this->isCompleted()) {
                $this->setCompleted();
            }
        } else {
            $this->reopen();
        }

        $isNew = $this->isNew();
        $result =  parent::save($data);

        if ($isNew) {
            App::services()->get('triggers')->trigger('object_created', $this);

            Singular_Event::dispatch('topic.create', $this);
        } else {
            Singular_Event::dispatch('topic.update', $this);
        }

        if ($moveCommentToTopic instanceof Comment) {
            $moveCommentToTopic->content = sprintf('%s [url=%s]%s[/url]', t('Moved to'), $this->getUrl(), $this->title);
            $moveCommentToTopic->save();
        }

        return $result;
    }

    /**
     * Returns true if user can leave comments
     *
     * @param System_Model_User $user
     *
     * @return boolean
     * @throws Exception
     */
    public function canCommented(System_Model_User $user)
    {
        return (!$this->lock_comments && $this->supportComments()) &&
            (
                (
                    $user->canAddComment() &&
                    $this->canSee($user) &&
                    !$user->isNew() &&
                    ($user->state == System_Mapper_State::NORMAL || $user->id == $this->created_by_id)
                ) ||
                (
                    $this->module === Topic::MODULE &&
                    $user->role()->isGuest() &&
                    Extra::feedbackAnonymousCommentsIsAllowed() &&
                    Singular_Core::_('Settings')->isOn('anonymous_topic_comments')
                )
            );
    }

    /**
     * Get Meta Description
     *
     * @return string
     */
    public function getMetaDescription()
    {
        $oldContent = $this->content;
        $this->content = Zend_Layout::getMvcInstance()->getView()->bbcode($this->content);
        $description = parent::getMetaDescription();
        $this->content = $oldContent;

        return $description;
    }

    /**
     * Converts current object's data to third-party applications using notation.
     *
     * @param ObjectInterface $dataSource
     * @return array
     */
    public function transform(ObjectInterface $dataSource = null)
    {
        if ($dataSource === null) {
            $dataSource = new SingleObject(Singular_Core::_('Auth')->getLoggedUser());
        }

        $data = parent::transform($dataSource);

        if (Features::isOn(Features::FEATURE_PROPERTIES)) {
            $data['custom_fields'] = $dataSource->customFields($this, 'feedback_topic');
        }

        return $data;
    }

    /**
     * Reverts user's vote by object
     *
     * @param System_Model_User $user
     * @param                   $value
     * @param ArrayObject       $cookie
     *
     * @return Resources_Model_Vote|null
     */
    public function setUserVote(System_Model_User $user = null, $value, $cookie)
    {
        parent::setUserVote($user, $value, $cookie);
        Feedback_Service_Trending::addScore($this, Feedback_Service_Trending::VOTE_BONUS, true);
    }

    public function fieldsForAgentForm($withProperties)
    {
        $data = parent::fieldsForAgentForm($withProperties);

        if (Features::isOn(Features::FEATURE_PROPERTIES) && $withProperties) {
            $properties = Singular_Loader::get('Properties')->findByAlias('feedback_topic', false, true);
            foreach ($properties as $property) {
                $value = $property->getValueForObject($this);

                if ($property->field_type === 'date') {
                    $value = System_Service_DateTime::dateForClient($value);
                }

                $data[Resources_Service_Properties::getName($property->id)] = $value;
            }
        }

        return $data;
    }

    /**
     * Return an array of value for template of email
     *
     * @param System_Model_User $user
     * @return array
     */
    public function getEmailValues(System_Model_User $user)
    {
        $view                      = Zend_Layout::getMvcInstance()->getView();
        $values                    = parent::getEmailValues($user);
        $values['object_id']       = Resources_Service_EmailTemplate::subjectIdSegment($this->id);
        $values['object_category'] = $this->getCategory() instanceof System_Model_Category ? $this->getCategory()->name : '-';
        $values['custom_fields']   = $user->role()->isAgent() ? $view->emailCustomFields('feedback_topic', $this) : '';
        $values['subject']         = $view->t('New ' . $this->getObjectType()->title, null, $this->forum()->language) . ' - "' . $view->escape($this->title) . '"';

        return $values;
    }
}
