<?php
/**************************************************************
 * Copyright UseResponse Inc., 2012-2016. All Rights Reserved.
 **************************************************************
 * NOTICE:  This source code is the property of UseResponse Inc.
 * In no case shall you rent, lease, lend, redistribute
 * any of below source code to a 3rd party individual or entity.
 *
 * @category   Helpdesk
 * @package    Helpdesk_Controller
 * @copyright  UseResponse Inc., 2012-2016 (https://www.useresponse.com)
 * @license    https://www.useresponse.com/eula_license-agreement (Commercial)
 */
use Useresponse\Application\Environment;

/**
 * Ticket controller
 *
 * @uses       System_Controller_Abstract_Action
 * @category   Helpdesk
 * @package    Helpdesk_Controller
 */
class Helpdesk_TicketController extends System_Controller_Abstract_Action
{
    /**
     * @var Helpdesk_Form_Ticket
     */
    public $formTicket;

    /**
     * @var Helpdesk_Model_Ticket
     */
    public $activeObject;

    /**
     * @var Resources_Model_DbTable_Files
     */
    protected $filesDB;

    /**
     * Initialize object
     *
     * Called from {@link __construct()} as final step of object instantiation.
     *
     * @see parent::init()
     */
    public function init()
    {
        parent::init();
        if ($id = $this->_getParam('id')) {
            $this->activeObject = Singular_Loader::get('Objects')->findById($id, $this->loggedUser);
        } elseif ($id = $this->_getParam('object_id')) {
            $this->activeObject = Singular_Loader::get('Objects')->findById($id, $this->loggedUser);
        } else {
            $this->activeObject = new Helpdesk_Model_Ticket();
        }

        if ($this->activeObject instanceof System_Model_Object) {
            $this->activeObject = $this->activeObject->convertObject();

            if ($this->activeObject->module !== 'helpdesk') {
                Singular_Core::_('Environment')->redirectToUrl(
                    $this->activeObject->getUrl()
                );
            }
        }

        if (!($this->activeObject instanceof Helpdesk_Model_Ticket)) {
            if ($this->loggedUser->isGuest()) {
                Singular_Core::_('Environment')->gotoLoginAndBackHere();
            } else {
                $this->notFound();
            }
        }

        /* Redirect to parent object for merged tickets */
        if ($this->activeObject->state == System_Mapper_State::MERGED) {
            $parent = $this->activeObject->getParentObject();

            if ($parent instanceof System_Model_Object) {
                $url = $parent->convertObject()->getUrl();
                $keyAdded = false;

                if (
                    ($share = Singular_Loader::get('Sharing')->findForUserAndObject($this->loggedUser, $this->activeObject)) &&
                    ($realUser = $share->getUser())
                ) {
                    if ($this->loggedUser->getAccessKey()) {
                        $realUser->setAccessKey($this->loggedUser->getAccessKey());
                    }

                    if ($realShare = Singular_Loader::get('Sharing')->findForUserAndObject($realUser, $parent)) {
                        $keyAdded = true;
                        $url .= "?access_key={$realShare->access_key}";
                    }
                }

                if (!$keyAdded && $this->loggedUser->getAccessKey()) {
                    $url .= "?access_key={$this->loggedUser->getAccessKey()}";
                }

                Singular_Core::_('Environment')->redirectToUrl($url);
            } else {
                $this->notFound();
            }
        }

        $this->filesDB            = new Resources_Model_DbTable_Files();
        $this->view->activeObject = $this->activeObject;

        $this->view->actionName = $this->getRequest()->getActionName();

        /* add/edit article */
        if (in_array($this->_getParam('action'), array('add', 'edit', 'objects', 'get', 'mobile-search'))) {
            /* check what are we add it is article or faq */
            $layout = $this->isMobileDevice() ? 'mobile' : 'layouts/ticket-add';
            $this->_helper->layout()->setLayout($layout);
            $formClass = $this->isMobileDevice() ? 'Helpdesk_Form_TicketMobile' : 'Helpdesk_Form_Ticket';
            $this->formTicket = new $formClass(array(
                'activeObject' => $this->activeObject,
                'post' => $this->getRequest()->getPost(),
            ));

            Resources_Service_Properties::injectForm($this->formTicket, $this->activeObject);

            if (!($this->activeObject->isNew() || $this->getRequest()->isPost())) {
                $this->formTicket->setDefaults($this->activeObject->toArray());
            }
            $this->view->form = $this->formTicket;
        }
    }

    /**
     * Mobile search action.
     *
     * @return void
     */
    public function mobileSearchAction()
    {
        if (!$this->isMobileDevice()) {
            Singular_Core::_('Environment')->redirectToRoute('ticket_add');
        }

        $this->formTicket->setDefaults(array(
            'title' => $this->_getParam('text')
        ));

        $this->view->layout()->setLayout('mobile');
    }

    /**
     * Add action
     *
     * @layout   layouts/ticket-add.phtml
     * @template helpdesk_ticket_add.phtml
     */
    public function addAction()
    {
        if ($this->isMobileDevice() && $this->loggedUser->isGuest() && Singular_Core::_('Settings')->isOff('anonymous_tickets')) {
            Singular_Core::_('Environment')->gotoLoginAndBackHere();
            return;
        }

        $this->formTicket->setDefaults(array(
            'title' => $this->_getParam('text')
        ));

        if (!empty($_COOKIE['ticket_description'])) {
            $this->formTicket->setDefaults(array(
                'content' => $_COOKIE['ticket_description']
            ));
            setcookie('ticket_description', '', time(), '/');
        }

        if ($this->getRequest()->isPost() || $this->getRequest()->isPut()) {
            if ($this->loggedUser->isGuest() && Singular_Core::_('Settings')->isOff('anonymous_tickets')) {
                $this->accessDenied();
            }
            $post = $this->getRequest()->getPost();

            /* @api */
            if ($this->api()->detected()) {
                $post['object_type'] = 'ticket';
                if (!array_key_exists('category_id', $post)) {
                    $post['category_id'] = 0;
                }
                if (!array_key_exists('is_private', $post)) {
                    $post['is_private'] = 0;
                }
                if (!array_key_exists('responsible_id', $post)) {
                    $post['responsible_id'] = 0;
                }
                if (!array_key_exists('file', $post)) {
                    $post['file'] = array();
                }
                if (!array_key_exists('featured', $post)) {
                    $post['featured'] = 1;
                }
            }

            if ($this->formTicket->isValid($post)) {
                $this->view->specifyEnabled = !empty($post['another_author']);
                $data = $this->formTicket->getValues();
                try {
                    $data['status_id'] = $this->activeObject->getOpenedStatus()->id;
                    if ($this->activeObject->save($data)) {
                        /* @api */
                        if ($this->api()->detected()) {
                            $this->api()->setSuccess($this->activeObject)->respondAndExit();
                        }
                        $redirectUrl = $this->activeObject->canSee($this->loggedUser)
                            ? $this->activeObject->getUrl()
                            : baseUrl();
                        $this->_redirect($redirectUrl, array(), 'Thank you for submitting request!');
                    } else {
                        $this->_redirect(baseUrl(), array(), 'Thank you for submitting request!');
                    }
                } catch (Exception $e) {
                    $this->notifyError($e->getMessage());
                }
            }
        } else {
            if ($this->loggedUser->role()->allowed('can_manage_tickets') && !$this->isMobileDevice()) {
                $this->formTicket->subscribers->setValue(array($this->loggedUser->id));
                Singular_AppClient::setParam('use-assignments', Singular_Core::_('Settings')->isOn('use_assignments'));
            }

            if ($categoryId = $this->_getParam('category_id')) {
                $this->formTicket->setDefault('category_id', $categoryId);
            }
        }

        Singular_AppClient::setParam('login.submit', Singular_Core::_('Settings')->isOn('anonymous_tickets') ? 1 : null);
    }

    /**
     * Edit action
     *
     * @layout   layouts/ticket-add.phtml
     * @template helpdesk_ticket_add.phtml
     */
    public function editAction()
    {
        if (!$this->activeObject->canEdit($this->loggedUser)) {
            if ($this->api()->detected()) {
                $this->api()->error(System_Service_Api::ERROR_FORBIDDEN);
            }

            Singular_Core::_('Environment')->accessDenied();
        }

        if ($this->getRequest()->isPost() || $this->getRequest()->isPut()) {
            $post = $this->getRequest()->getPost();

            /* @api */
            if ($this->api()->detected()) {
                $post['object_type'] = 'ticket';
                if (!array_key_exists('category_id', $post)) {
                    $post['category_id'] = $this->activeObject->category_id;
                }
                if (!array_key_exists('title', $post)) {
                    $post['title'] = $this->activeObject->title;
                }
                if (!array_key_exists('is_private', $post)) {
                    $post['is_private'] = 0;
                }
                if (!array_key_exists('content', $post)) {
                    $post['content'] = $this->activeObject->content;
                }
                if (!array_key_exists('status_id', $post)) {
                    $post['status_id'] = $this->activeObject->status_id;
                }
                if (!array_key_exists('responsible_id', $post)) {
                    $post['responsible_id'] = $this->activeObject->responsible_id;
                }
                if (!array_key_exists('file', $post)) {
                    $post['file'] = array();
                }
                if (!array_key_exists('featured', $post)) {
                    $post['featured'] = 1;
                }
                if (array_key_exists('status', $post)) {
                    $newStatus = Singular_Loader::get('Statuses')->findBySlug($post['status'], 'helpdesk');
                    if ($newStatus instanceof System_Model_Status && $newStatus->object_type === 'ticket') {
                        $post['status_id'] = $newStatus->id;
                    }
                    unset($post['status']);
                }
            }

            if ($this->formTicket->isValid($post)) {
                $data = $this->formTicket->getValues();
                $this->activeObject->save($data);
                if (Singular_AppClient::getParam('canUseAttachments') > 0) {
                    $this->filesDB->saveAttachments($this->_getParam('fileinfo'), $this->activeObject);
                }
                $this->filesDB->saveWysiwygImages($this->activeObject);
                /* @api */
                if ($this->api()->detected()) {
                    $this->api()->setSuccess($this->activeObject)->respondAndExit();
                }
                $this->_redirect($this->activeObject->getUrl(), array(), 'Ticket is updated!');
            }
        } else {
            if (!$this->isMobileDevice()) {
                if ($this->loggedUser->role()->allowed('can_manage_tickets')) {
                    $this->formTicket->subscribers->setValue(
                        Singular_Loader::get('Subscribers')->getUserIdsByObject($this->activeObject)
                    );
                    $this->formTicket->responsible_id->setValue($this->activeObject->responsible_id);
                    $this->formTicket->submit_ticket->setLabel('Save Changes');
                }

                $this->title = $this->view->t('Edit Ticket');
            }

            if (Singular_AppClient::getParam('canUseAttachments') > 0) {
                $this->formTicket->getSubForm('file')->setAttachments($this->activeObject);
            }
            /* @api */
            if ($this->api()->detected()) {
                $this->api()->error(System_Service_Api::ERROR_CONFLICT);
            }
            $template = $this->isMobileDevice() ? 'helpdesk_ticket_add.mobile.phtml' : 'helpdesk_ticket_add.phtml';
            $this->renderScript($template);
            Singular_AppClient::setParam('use-assignments', Singular_Core::_('Settings')->isOn('use_assignments'));
        }
    }

    /**
     * Reload Captcha action
     *
     */
    public function reloadCaptchaAction()
    {
        if ($this->isAjax() && $this->getRequest()->isPost()) {
            $form                = new Helpdesk_Form_Ticket();
            $captcha             = $form->getElement('captcha')->getCaptcha();
            $captchaData         = array();
            $captchaData['id']   = $captcha->generate();
            $captchaData['src']  = $captcha->getImgUrl() .
                $captcha->getId() .
                $captcha->getSuffix();
            $this->view->captcha = $captchaData;
        } else {
            throw new Zend_Exception('Bad Request!');
        }
    }

    /**
     * List action
     *
     * @layout   layouts/article-list.phtml
     * @template knowledge-base_ticket_list.phtml
     */
    public function listAction()
    {
        if (!$this->loggedUser->role()->allowed('can_manage_tickets')) {
            Singular_Core::_('Environment')->accessDenied();
        }

        $template = $this->isMobileDevice() ? 'mobile' : 'search';
        $this->view->layout()->setLayout($template);
        $where = array(
            'module'         => Helpdesk_Model_Ticket::MODULE,
            'date'           => $this->_getParam('date'),
            'text'           => $this->_getParam('text'),
            'state'          => System_Mapper_State::DEFAULT_STATE,
            'category_id'    => $this->_getParam('category_id', 'all'),
            'completed_on'   => 'all',
            'responsible_id' => $this->_getParam('responsible_id', 'all'),
            'order'          => $this->_getParam('order', 'new_updated'),
            'status_id'      => $this->_getParam('status_id', 'all-active'),
            'status'         => $this->_getParam('status', $this->isApiCall() ? 'all-active' : 'all'),
        );

        if ($this->isApiCall()) {
            unset($where['status_id']);
            if ($authorId = $this->getRequest()->getParam('authorId')) {
                $where['created_by_id'] = (int)$authorId;
            }
        }

        if (!is_null($this->view->ticketSearch = $this->_getParam('ticket_search')) && $this->view->ticketSearch) {
            $where['object_key'] = preg_replace('/^#/', '', trim($this->view->ticketSearch));

            if (preg_match('/^tag:/i', $where['object_key'])) {
                $where['status_id'] = 'all';
            }

            $where['state'] = array(System_Mapper_State::NORMAL);
            $where['completed_on'] = 'all';
        }

        $page = isset($_GET['page']) ? $_GET['page'] : $this->_getParam('page', 1);
        $paginator = Singular_Loader::get('Tickets')->getPaginator($where, $page, null, $this->_getParam('count'), $this->loggedUser);

        if ($this->isApiCall()) {
            $this->api()->setSuccess($paginator)->respond(true);
        }

        $this->view->assign(
            array(
                'paginator'  => $paginator,
                'categories' => Singular_Loader::get('Categories')->findAllByUser(
                        $this->loggedUser, Helpdesk_Model_Ticket::MODULE
                    ),
                'agents'     => Singular_Loader::get('Users')->findByPermission('can_manage_tickets'),
                'statuses'   => Singular_Loader::get('Statuses')->findByObjectType('ticket', 'helpdesk'),
                'activeState' => $this->_getParam('completed_on', 'open'),
                'activeStatus' => $this->_getParam('status_id', 'all'),
                'activeCategory' => $this->_getParam('category_id', 'all'),
                'activeResponsible' => $this->_getParam('responsible_id', 'all'),
                'activeSort' => $this->_getParam('order', 'new_updated'),
            )
        );

        if (!empty($where['object_key']) && $this->view->paginator->getTotalItemCount() == 0) {
            $where['text'] = $where['object_key'];
            unset($where['object_key']);
            $this->view->paginator = Singular_Loader::get('Tickets')->getPaginator(
                $where, $page, null, null, $this->loggedUser
            );
        }

        $pages = ceil($this->view->paginator->getTotalItemCount() / $this->view->paginator->getItemCountPerPage());

        if ($this->isAjax()) {
            $template = $this->isMobileDevice() ? '_tickets_list.mobile.phtml' : 'helpdesk_ticket_list.phtml';
            $this->autoresponder->setOutput($this->fetchTemplate($template));
            $this->autoresponder->setParam('pages', $pages);
            $this->autoresponder->setParam('totalCount', (string)$this->view->paginator->getTotalItemCount());
        } elseif (!$this->isMobileDevice()) {
            $this->pageStructure->setContentBeforeSidebar($this->view->render('_ticket_search_left_block.phtml'))
                ->setContentAfterSidebar($this->view->render('ctrl-help.phtml'))
                ->appendFilter($this->view->render('_tickets_filter.phtml'))
                ->setFilterAreaButton($this->view->render('_tickets_filterarea_buttons.phtml'))
                ->setTitle($this->view->t('All Tickets'));

            $this->view->additionalModulePath('system');
            $this->pageStructure->setSidebar(
                $this->view->render('_block_objects-navigation.phtml')
            );

            $_massEditForm            = new Helpdesk_Form_MassEdit();
            $this->view->massEditForm = $_massEditForm->render();
            $this->pageStructure->appendSidebar($this->view->render('_mass_edit_form.phtml'));
        } else {
            Singular_AppClient::setParam('pages', $pages);
        }
    }

    /**
     * Profile list action
     *
     * @return void
     * @throws Singular_Acl_Exception
     */
    public function profileListAction()
    {
        if (!$this->isMobileDevice()) {
            $this->invalidRequest();
        }

        $this->view->activeUser = Singular_Loader::get('Users')->findById($this->_getParam('user_id'));

        if (!($this->view->activeUser instanceof System_Model_User)) {
            $this->notFound();
        }

        $this->view->layout()->setLayout('mobile');
        $this->view->additionalModulePath('system');
        $this->view->paginator = Singular_Loader::get("Tickets")->getPaginator(array(
            'module'        => Helpdesk_Model_Ticket::MODULE,
            'created_by_id' => $this->view->activeUser->id,
            'order'         => 'new_updated'
        ), $this->_getParam('page', 1));

        $pages = ceil($this->view->paginator->getTotalItemCount() / $this->view->paginator->getItemCountPerPage());

        if ($this->isAjax()) {
            $this->autoresponder->setOutput($this->fetchTemplate('_tickets_list.mobile.phtml'));
            $this->autoresponder->setParam('pages', $pages);
        } else {
            Singular_AppClient::setParam('pages', $pages);
        }
    }

    /**
     * Show action
     *
     * @layout   layouts/ticket.phtml
     * @template helpdesk_ticket_show.phtml
     */
    public function showAction()
    {
        if (!$this->activeObject->canSee($this->loggedUser)) {
            Singular_Core::_('Environment')->accessDenied();
        }

        $layout = $this->isMobileDevice() ? 'mobile' : 'layouts/ticket';
        $this->_helper->layout()->setLayout($layout);
        // Initialize comment form
        $form = new Resources_Form_CommentAdd([
            'activeObject' => $this->activeObject
        ]);
        $form->setActiveObject($this->activeObject);
        $form->setDefault('object_id', $this->activeObject->id);
        $form->setDefault('object_name', $this->activeObject->getObjectName());
        Singular_AppClient::setParam('status_id', $this->activeObject->status_id);
        Singular_AppClient::setParam('state', $this->activeObject->isCompleted());
        Singular_AppClient::setParam('leave-comment', $this->_getParam('leave-comment'));
        Singular_AppClient::setParam('use-assignments', Singular_Core::_('Settings')->isOn('use_assignments'));

        $this->view->commentsAddForm = $form;

        if ($this->isMobileDevice()) {
            if ($this->activeObject->canEdit($this->loggedUser)) {
                $_massEditForm = new Helpdesk_Form_MassEditMobile(array('activeObject' => $this->activeObject));
                $this->view->massEditForm = $_massEditForm->render();
            }

            Singular_Runtime::store('activeCannedResponsesType', 'ticket');
        }

        Singular_View::useSiteTitle(false);

        if ($this->loggedUser->getAccessKey()) {
            Singular_AppClient::setParam('commentingByKey', 1);
        }
    }

    /**
     * User info action.
     *
     * @template _ticket_user_info.phtml
     */
    public function userInfoAction()
    {
        if (!$this->isAjax()) {
            $this->invalidRequest();
        }

        if (!$this->activeObject->canSee($this->loggedUser)) {
            $this->accessDenied();
        }

        $this->autoresponder->setOutput($this->fetchTemplate('_ticket_user_info.phtml'));
    }

    /**
     * Complete action
     *
     * @layout   main.phtml
     * @template .phtml
     */
    public function completeAction()
    {
        $previousStatus = $this->activeObject->getStatusName();
        $previousStatusTpl = $this->activeObject->statusEmailTemplate();

        if ((bool)$this->_getParam('is_completed')) {
            $this->activeObject->status_id = $this->activeObject->getOpenedStatus()->id;
        } else {
            $this->activeObject->status_id = $this->activeObject->getCompletedStatus()->id;
        }

        $this->activeObject->save();

        if ($this->isAjax()) {
            $this->autoresponder
                ->setSuccess($this->activeObject->isCompleted() ? 'Ticket is completed!' : 'Ticket is reopened!')
                ->setParam(
                    'href', $this->_helper->url->url(
                        array(
                            'id'         => $this->activeObject->id,
                            'is_completed' => $this->activeObject->isCompleted()
                        ), 'ticket_toggle_complete'
                    )
                )
                ->setParam(
                    'title',
                    $this->view->t($this->activeObject->isCompleted() ? 'Reopen' : 'Complete')
                )
                ->setParam('changeStateCheckboxCaption', (
                    $this->view->t($this->activeObject->isCompleted() ? 'Reopen Ticket' : 'Complete Ticket')
                ))
                ->setParam('status_id', $this->activeObject->status_id);

            $status = $this->activeObject->getStatus();
            $this->autoresponder->setParam(
                'statusParam', array(
                    'color'      => $status->color,
                    'text_color' => $status->text_color,
                    'title'      => $this->view->t($status->title)
                )
            );

            if (Singular_Core::_('Settings')->isOn('tickets_completion_notifications') || (int)$this->activeObject->getStatus()->is_closed == 0) {
                $users = Singular_Loader::get('Subscribers')->findUsersByObject($this->activeObject);
                $keys = $this->activeObject->getEmailValues();
                $keys['previous_status'] = $previousStatus;
                $keys['previous_status_tpl'] = $previousStatusTpl;

                foreach ($users as $user) {
                    $keys['object_url'] = $this->activeObject->getEmailUrl($user);
                    $keys['unsubscribe_url'] = $this->activeObject->getUnsubscribeUrl($user);
                    Resources_Service_ApplicationMailer::send(
                        $user, 'change_object_status', 'resources', $keys, $this->activeObject
                    );
                }
            }

        } else {
            Singular_Core::_('Environment')->redirectToUrl($this->activeObject->getUrl());
        }
    }

    /**
     * Change-status action
     *
     * Changes status for specified response
     *
     * @return void
     */
    public function changeStatusAction()
    {
        if (!$this->activeObject->canManage($this->loggedUser)) {
            return;
        }

        $statusDb  = Singular_Loader::get('Statuses');
        $validator = new Zend_Validate_Db_RecordExists(array(
            'table' => $statusDb->getTableName(),
            'field' => 'id'
        ));

        $validator->getSelect()->where('object_type = ?', $this->activeObject->object_type)->where(
            'module = ?', $this->activeObject->module
        );
        $statusId = (int)$this->_getParam('status_id');
        if ($validator->isValid($statusId)) {
            $this->activeObject->switchStatus($statusId);
            $this->autoresponder->setSuccess(
                $this->view->t(
                    ':module status has been updated!',
                    array('module' => $this->view->t(ucfirst($this->activeObject->getType()->title)))
                )
            );

            $res          = $statusDb->getAdapter()->fetchRow(
                "SELECT `title` ,`color`, `text_color` FROM `{$statusDb->getTableName()}` WHERE `id` = ?",
                array($statusId)
            );
            $res['title'] = $this->view->t($res['title']);
            $this->autoresponder
                ->setOutput(json_encode($res))
                ->setParam('changeStateCheckboxCaption', (
                $this->view->t($this->activeObject->isCompleted() ? 'Reopen Ticket' : 'Complete Ticket')
                ))
                ->setParam('complete_href', $this->view->url(
                    array(
                        'id'           => $this->activeObject->id,
                        'is_completed' => $this->activeObject->isCompleted()
                    ), 'ticket_toggle_complete'
                ))
                ->setParam('complete_title', $this->view->t($this->activeObject->isCompleted() ? 'Reopen' : 'Complete'));
        }
    }

    /**
     * Search action
     *
     * @layout   layouts/article-list.phtml
     * @template knowledge-base_ticket_search.phtml
     */
    public function searchAction()
    {
        if (!$this->loggedUser->role()->allowed('can_manage_tickets')) {
            Singular_Core::_('Environment')->accessDenied();
        }

        $this->_helper->layout()->setLayout('layouts/article-list');
        $this->view->assign(
            array(
                'searchPhrase' => $this->_getParam('search_phrase'),
                'articles'     => Singular_Loader::get('Articles')->findByParams(
                        $this->_getAllParams(), $this->loggedUser
                    )
            )
        );
    }

    /**
     * Vote-add action
     *
     * Adds new vote to specified objects
     */
    public function voteAddAction()
    {
        $this->_setParam('output_helper_name', 'articleVote');
        $this->_helper->actionStack(
            'vote-toggle',
            'object',
            'system',
            array($this->_getAllParams())
        );
    }

    /**
     * Import-emails action
     *
     * Import emails from active mailbox
     */
    public function importEmailsAction()
    {
        if (!$this->loggedUser->role()->allowed('can_manage_tickets') || Environment::isOnCloud()) {
            Singular_Core::_('Environment')->accessDenied();
        }

        $this->api()->acceptMethods(['GET']);

        HelpDesk_Service_MailProcessor::importEmails();

        if ($this->isApiCall()) {
            $this->api()->setSuccess();
        } elseif ($this->isAjax()) {
            $this->autoresponder->setSuccess('Tickets are updated!');
        }
    }

    /**
     * Mass edit action
     *
     * @return void
     */
    function massEditAction()
    {
        if ($this->isAjax() && $this->_request->isPost()) {
            $post = $this->getRequest()->getPost();
            $ticketIds = explode(",", $post['tickets']);
            $tickets = Singular_Loader::get("Tickets")->findByIds($ticketIds, $this->loggedUser);
            $aRes = $this->view;
            $res = function () use ($ticketIds, $aRes) {
                Singular_Core::_('Cache')->removeAllByTags(['helpdesk']);
                $aRes->notify()->success($aRes->t(':count item(s) are updated!', array('count' => count($ticketIds))));
            };

            if (!empty($post['merge'])) {
                if (count($ticketIds) > 1) {
                    $objects = Singular_Loader::get('Objects');
                    $leadObject = $objects->fetchRow(
                        $objects->select()->where('id in (?)', $ticketIds)->order('created_at')
                    );

                    if ($leadObject instanceof System_Model_Object) {
                        unset($ticketIds[array_search($leadObject->id, $ticketIds)]);
                        $leadObject = $leadObject->convertObject();
                        Resources_Service_Merger::merge($leadObject, $ticketIds);
                    }
                }

                $res();
                return;
            }

            if(!empty($post['properties']['state'])){
                $state = $post['properties']['state'];
                foreach ($tickets as $ticket) {
                    /**@var $ticket Helpdesk_Model_Ticket */
                    if ($state == System_Mapper_State::TRASHED) {
                        System_Mapper_State::object($ticket)->moveToTrash();
                    } elseif ($state == System_Mapper_State::NORMAL) {
                        System_Mapper_State::object($ticket)->recoverFromTrash();
                    }
                }
                $res();
                return;
            }

            $previousStatuses = array();
            $previousStatusesTpl = array();

            if (isset($post['properties']) && is_array($post['properties'])) {

                /** @var Helpdesk_Model_Ticket $ticket */
                foreach ($tickets as $ticket) {
                    if (isset($post['properties']['status_id']) && (int)$post['properties']['status_id'] !== (int)$ticket->getStatus()->id) {
                        $previousStatuses[$ticket->id] = $ticket->getStatusName();
                        $previousStatusesTpl[$ticket->id] = $ticket->statusEmailTemplate();
                    }
                    $ticket->save($post['properties']);
                }
            }

            if (isset($post['properties']['status_id'])) {
                foreach ($tickets as $ticket) {
                    /**@var $ticket Helpdesk_Model_Ticket*/
                    if (
                        !isset($previousStatuses[$ticket->id]) ||
                        Singular_Core::_('Settings')->isOff('tickets_completion_notifications') && (int)$ticket->getStatus()->is_closed == 1
                    ) {
                        continue;
                    }
                    $users = Singular_Loader::get('Subscribers')->findUsersByObject($ticket);
                    $keys = $ticket->getEmailValues();
                    $keys['previous_status'] = $previousStatuses[$ticket->id];
                    $keys['previous_status_tpl'] = $previousStatusesTpl[$ticket->id];
                    foreach ($users as $user) {
                        $keys['unsubscribe_url'] = $ticket->getUnsubscribeUrl($user);
                        $keys['object_url'] = $ticket->getEmailUrl($user);
                        Resources_Service_ApplicationMailer::send($user, 'change_object_status', 'resources', $keys, $ticket);
                    }
                }
            }
            $res();
            return;
        }
    }
}
