<?php
/**************************************************************
 * Copyright UseResponse Inc., 2012-2016. All Rights Reserved.
 **************************************************************
 * NOTICE:  This source code is the property of UseResponse Inc.
 * In no case shall you rent, lease, lend, redistribute
 * any of below source code to a 3rd party individual or entity.
 *
 * @category   System
 * @package    System_Model
 * @subpackage System_Model_DbTable
 * @copyright  UseResponse Inc., 2012-2016 (https://www.useresponse.com)
 * @license    https://www.useresponse.com/eula_license-agreement (Commercial)
 */

/**
 * "<prefix>objects" table model
 *
 * @uses       System_Model_DbTable_Abstract
 * @category   System
 * @package    System_Model
 * @subpackage System_Model_DbTable
 */
class System_Model_DbTable_Objects extends System_Model_DbTable_Abstract
{
    /**
     * Default fetch limit
     */
    const LIMIT = 10;

    /**
     * Type 'any' means all of object types
     */
    const module_ANY = 'any';

    /**
     * The table name
     *
     * @var string
     */
    protected $_name = 'objects';

    /**
     * The primary key column or columns.
     * A compound key should be declared as an array.
     * You may declare a single-column primary key
     * as a string.
     *
     * @var mixed
     */
    protected $_primary = 'id';

    /**
     * Row class
     *
     * @var string
     */
    protected $_rowClass = 'System_Model_Object';

    /**
     * Dependent Tables
     *
     * @var array
     */
    protected $_dependentTables
        = array(
            'System_Model_DbTable_Subscribers',
            'Resources_Model_DbTable_Comments',
            'Resources_Model_DbTable_Votes',
        );

    /**
     * Reference map
     *
     * @var array
     */
    protected $_referenceMap = array(
        'Status'     => array(
            'columns'       => array('status', 'module'),
            'refTableClass' => 'System_Model_DbTable_Statuses',
            'refColumns'    => array('slug', 'module'),
            'onDelete'      => self::CASCADE,
        ),
        'CreatedBy'  => array(
            'columns'       => array('created_by_id'),
            'refTableClass' => 'System_Model_DbTable_Users',
            'refColumns'    => array('id'),
            'onDelete'      => self::CASCADE,
        ),
        'File'       => array(
            'columns'       => array('id'),
            'refTableClass' => 'Resources_Model_DbTable_Files',
            'refColumns'    => array('object_id')
        ),
        'Categories' => array(
            'columns'       => array('category_id', 'module'),
            'refTableClass' => 'System_Model_DbTable_Categories',
            'refColumns'    => array('id', 'module'),
        )
    );

    protected $_typesArray = null;

    /**
     * Inserts a new object row.
     *
     * @see    parent::insert()
     *
     * @param  array $data Column-value pairs.
     *
     * @return mixed         The primary key of the row inserted.
     */
    public function insert(array $data)
    {
        $data = $this->_setDate($data, 'created_at');
        $data = $this->_setActor($data, 'created_by_id');

        /* try-catch wrapper for spam bots (when created_by_id is null) */
        try {
            return parent::insert($data);
        } catch (Exception $e) {
            Singular_Core::_('Environment')->invalidRequest();
        }
    }

    /**
     * Returns objects by given criteria.
     *
     * @param  array|null           $where
     * @param  string|array         $order
     * @param  int                  $limit
     * @param  Zend_Db_Table_Select $select
     *
     * @return Zend_Db_Table_Rowset
     */
    public function findAll($where = null, $order = null, $limit = null, $select = null)
    {
        if (!$select instanceof Zend_Db_Table_Select) {
            $select = $this->getSelect();
        }

        $tableName = $this->getTableName();
        if (is_array($where) && isset($where['status']) && $where['status'] == 'new') {
            $select->where("$tableName.created_at > ?", date("Y-m-d H:i:s", strtotime("-7 day last Monday")));
        }
        if ($order) {
            $select->order($order);
        } else {
            $select->order("created_at DESC");
        }
        if ($limit) {
            $select->limit($limit);
        }

        return $this->fetchAll($select);
    }

    /**
     * Returns prepared select object
     *
     * @see    parent::getSelect()
     * @return Zend_Db_Table_Select
     */
    public function getSelect()
    {
        $select = parent::getSelect();
        if (System_Service_Environment::getInstance()->isAdmin()) {
            $select->where("{$this->getTableName()}.state <> 0");
        } else {
            $select->where("{$this->getTableName()}.state=?", System_Mapper_State::DEFAULT_STATE);
        }

        return $select;
    }

    /**
     * Return fetched paginator rows.
     *
     * @param  array                  $where
     * @param  int                    $page
     * @param  mixed                  $order
     * @param  int                    $limit
     * @param  System_Model_User|null $loggedUser
     *
     * @return Zend_Paginator
     */
    public function getPaginator($where, $page, $order = null, $limit = null, System_Model_User $loggedUser = null)
    {
        $select = $where;

        if (!$where instanceof Zend_Db_Select) {
            if (empty($where['state'])) {
                $select = $this->getSelect();
            } else {
                if (!is_array($where['state'])) {
                    $where['state'] = array($where['state']);
                }

                $select = parent::getSelect()->where("{$this->getTableName()}.state IN (?)", $where['state']);
            }

            $select->distinct(true);
            $select = $this->prepareSelect($select, $where, $order, $loggedUser);
        }

        if (empty($limit)) {
            $limit = Singular_Core::_('Settings')->get('paginator_item_count_per_page.value');
        }

        return $this->getPaginatorRows($select, null, null, $page, $limit);
    }

    /**
     * Prepares select for responses fetching
     *
     * @param  Zend_Db_Select    $select
     * @param  array             $where
     * @param  mixed             $order
     * @param  System_Model_User $loggedUser
     *
     * @return Zend_Db_Select
     * @throws Zend_Db_Exception
     */
    public function prepareSelect($select, $where, $order = null, System_Model_User $loggedUser = null)
    {
        if (!($select instanceof Zend_Db_Select)) {
            throw new Zend_Db_Exception('$select must be instance of Zend_Db_Select');
        }
        $tableName = $this->getTableName();
        $objOwner  = System_Model_Object::OWNER;
        /* date */
        if (!empty($where['start_date'])) {
            $select->where("$tableName.created_at >= ?", date("Y-m-d H:i:s", strtotime($where['start_date'])));
        }
        if (!empty($where['end_date'])) {
            $select->where(
                "$tableName.created_at <= ?", date("Y-m-d H:i:s", strtotime($where['end_date'] . "24:00:00"))
            );
        }

        if (array_key_exists('is_private', $where)) {
            $seePrivate = (int)(bool)$where['is_private'];
            $select->where("$tableName.is_private <= {$seePrivate}");
        } elseif ($loggedUser) {
            $userId     = $loggedUser->isGuest() ? 0 : $loggedUser->id;
            $seePrivate = (int)$loggedUser->role()->isAgent();
            $select->where("$tableName.is_private <= {$seePrivate} OR $tableName.created_by_id = $userId");
        }

        if (!empty($where['date']) && $where['date'] != 'all') {
            switch ($where['date']) {
                case 'today' :
                    $select->where(
                        "$tableName.created_at > ?",
                        date("Y-m-d H:i:s", strtotime("-1 day"))
                    );
                    break;
                case 'yesterday' :
                    $select->where(
                        "$tableName.created_at > ?",
                        date("Y-m-d H:i:s", strtotime("-2 day"))
                    );
                    $select->where(
                        "$tableName.created_at < ?",
                        date("Y-m-d H:i:s", strtotime("-1 day"))
                    );
                    break;
                case 'this_week' :
                    $select->where(
                        "$tableName.created_at > ?",
                        date("Y-m-d H:i:s", strtotime("last Monday"))
                    );
                    break;
                case 'last_week' :
                    $select->where(
                        "$tableName.created_at > ?",
                        date("Y-m-d H:i:s", strtotime("-7 day last Monday"))
                    );
                    $select->where(
                        "$tableName.created_at < ?",
                        date("Y-m-d H:i:s", strtotime("last Monday"))
                    );
                    break;
                case 'this_month' :
                    $select->where(
                        "$tableName.created_at > ?",
                        date("Y-m-d H:i:s", mktime("00", "00", "00", date('m'), "01", date('Y')))
                    );
                    break;
                case 'last_month' :
                    $time = strtotime("last month");
                    $select->where(
                        "$tableName.created_at > ?",
                        date("Y-m-d H:i:s", mktime("00", "00", "00", date('m', $time), "01", date('Y', $time)))
                    );
                    $select->where(
                        "$tableName.created_at < ?",
                        date("Y-m-d H:i:s", mktime("00", "00", "00", date('m'), "01", date('Y')))
                    );
                    break;
            }
        }
        if (!empty($where['created_by_id'])) {
            $sharingForUser = Singular_Loader::get('Users')->findById($where['created_by_id']);
            $sharingCondition = null;
            if (!empty($where['and_shared']) && $sharingForUser instanceof System_Model_User && !$sharingForUser->role()->isAgent()) {
                $select->joinLeft(['sharing' => Singular_Loader::get('Sharing')->getTableName()], "sharing.object_id = {$tableName}.id", []);
                $sharingCondition .= Singular_Core::_('Db')->getAdapter()->quoteInto('or sharing.email = ?', $sharingForUser->email);
            }
            $select->where("$tableName.created_by_id=? {$sharingCondition}", $where['created_by_id'], Zend_Db::INT_TYPE);
        }
        if (array_key_exists('object_type', $where) && $where['object_type'] != 'all') {
            if (empty($where['object_type'])) {
                $select->where(0);
            } else {
                $select->where("$tableName.object_type IN (?)", $where['object_type']);
            }
        }

        if (array_key_exists('active', $where) && $where['active']) {
            $oT = Singular_Loader::get('ObjectTypes')->getTableName();
            $select->join(array('ot' => $oT), "ot.slug = {$this->getTableName()}.object_type AND ot.is_active = 1",
                array());
        }

        if (!empty($where['exclude_id'])) {
            $where['exclude_id'] = is_array($where['exclude_id']) ? $where['exclude_id'] : (int)$where['exclude_id'];
            $select->where("$tableName.id NOT IN (?)", $where['exclude_id']);
        }

        /* add object object_type in where conditions */
        if (!empty($where['module'])) {
            if (is_array($where['module'])) {
                $select->where("$tableName.module IN (?)", $where['module']);
            } else {
                $select->where("$tableName.module = ?", $where['module']);
            }
        }

        if (!empty($where['assignment']) && $where['assignment'] !== 'all') {
            if ($where['assignment'] == 'selected-filter') {
                if (is_array($where['responsible_id'])) {
                    if (isset($where['assignment-type']) && $where['assignment-type'] == 'subscribed') {
                        $sT = Singular_Loader::get('Subscribers')->getTableName();
                        $select->where("(SELECT COUNT(*) FROM {$sT} WHERE object_id = {$tableName}.id AND user_id IN (?)) > 0",
                            $where['responsible_id']);
                        $where['responsible_id'] = 'all';
                    }
                } else {
                    $where['responsible_id'] = 'all';
                }
            } elseif ($where['assignment'] == 'me') {
                if (isset($where['assignment-type']) && $where['assignment-type'] == 'subscribed') {
                    $sT = Singular_Loader::get('Subscribers')->getTableName();
                    $select->where("(SELECT COUNT(*) FROM {$sT} WHERE object_id = {$tableName}.id AND user_id = '" . $loggedUser->id . "') > 0");
                } else {
                    $select->where("$tableName.responsible_id = ?", (int)$loggedUser->id, Zend_Db::INT_TYPE);
                }
            } else {
                $where['responsible_id'] = $where['assignment'];
            }
        }

        if (!empty($where['responsible_id']) && $where['responsible_id'] !== 'all') {
            switch ($where['responsible_id']) {
                case 'set':
                    $select->where("$tableName.responsible_id <> ?", 0, Zend_Db::INT_TYPE);
                    $uT = Singular_Core::_('Db')->getTablePrefix('users');
                    $select->where("(SELECT `state` FROM {$uT} ut WHERE ut.id = $tableName.responsible_id) = ?",
                        System_Model_User::STATE_NORMAL);
                    break;
                case 'not-set':
                    $select->where("$tableName.responsible_id = ?", 0, Zend_Db::INT_TYPE);
                    break;
                case 'me':
                    $select->where("$tableName.responsible_id = ?", (int)$loggedUser->id, Zend_Db::INT_TYPE);
                    break;
                default:
                    $where['responsible_id'] = is_array($where['responsible_id']) ? $where['responsible_id'] : (int)$where['responsible_id'];
                    $select->where("$tableName.responsible_id IN (?)", $where['responsible_id']);
            }
        }

        if (isset($where['category']) && $where['category'] != 'all') {
            switch ($where['category']) {
                case 'with-category';
                    $select->where('category_id > 0');
                    break;
                case 'no-category':
                    $select->where('category_id = 0');
                    break;
                default:
                    if (is_array($where['category'])) {
                        $select->where('category_id IN (?)', $where['category']);
                    }
            }
        }

        if (isset($where['category_slug'])) {
            if ($where['category_slug'] == System_Model_Category::UNCATEGORIZED_SLUG
                || ($cat = Singular_Loader::get('Categories')->findBySlug($where['category_slug']))
            ) {
                $catId = $where['category_slug'] == System_Model_Category::UNCATEGORIZED_SLUG
                    ? 0 : $cat->id;
                $select->where("$tableName.category_id = ?", $catId, Zend_Db::INT_TYPE);
            }
        }

        if (isset($where['completed_on']) && !in_array($where['completed_on'], ['all', 'open' ,'completed'], true)) {
            $select->where("$tableName.completed_on IS NOT NULL");

            switch ($where['completed_on']) {
                case 'today':
                    $select->where(
                        "$tableName.completed_on > ?",
                        date("Y-m-d H:i:s", strtotime("-1 day"))
                    );
                    break;
                case 'yesterday' :
                    $select->where(
                        "$tableName.completed_on > ?",
                        date("Y-m-d H:i:s", strtotime("-2 day"))
                    );
                    $select->where(
                        "$tableName.completed_on < ?",
                        date("Y-m-d H:i:s", strtotime("-1 day"))
                    );
                    break;
                case 'this_week' :
                    $select->where(
                        "$tableName.completed_on > ?",
                        date("Y-m-d H:i:s", strtotime("last Monday"))
                    );
                    break;
                case 'last_week' :
                    $select->where(
                        "$tableName.completed_on > ?",
                        date("Y-m-d H:i:s", strtotime("-7 day last Monday"))
                    );
                    $select->where(
                        "$tableName.completed_on < ?",
                        date("Y-m-d H:i:s", strtotime("last Monday"))
                    );
                    break;
                case 'this_month' :
                    $select->where(
                        "$tableName.completed_on > ?",
                        date("Y-m-d H:i:s", mktime("00", "00", "00", date('m'), "01", date('Y')))
                    );
                    break;
                case 'last_month' :
                    $time = strtotime("last month");
                    $select->where(
                        "$tableName.completed_on > ?",
                        date("Y-m-d H:i:s", mktime("00", "00", "00", date('m', $time), "01", date('Y', $time)))
                    );
                    $select->where(
                        "$tableName.completed_on < ?",
                        date("Y-m-d H:i:s", mktime("00", "00", "00", date('m'), "01", date('Y')))
                    );
                    break;
            }
        }

        /* expires */
        if (!empty($where['check_expires'])) {
            $select->where("$tableName.expires_at > ? or $tableName.expires_at is null", date('Y-m-d'));
        }

        /* order */
        if (
            empty($where['order']) ||
            $where['order'] == 'all' ||
            (
                $where['order'] === 'trending' &&
                (!Singular_Module::getInstance()->isActive('feedback') || !Feedback_Service_Trending::isWorkable())
            )
        ) {
            $where['order'] = 'popular';
        }
        if (!empty($where['order'])) {
            switch ($where['order']) {
                case 'new':
                    if (empty($order)) {
                        $order['order'] = 'created_at';
                        $order['sort']  = 0;
                    }
                    break;
                case 'updated':
                    if (empty($order)) {
                        $order['order'] = 'commented_at';
                        $order['sort']  = 0;
                    }

                    break;
                case 'new_updated':
                    if (empty($order)) {
                        $order['order'] = 'updated';
                    }
                    break;
                case 'trending':
                    if (empty($order)) {
                        $order = [
                            'order' => 'score',
                            'sort' => 0,
                        ];
                    }
                    break;
                default:
                    $order = [
                        'order' => 'votes',
                        'sort' => 0,
                    ];
                    break;
            }
        }

        if (isset($where['status']) && $where['status'] && $where['status'] != 'all') {
            $statusT = Singular_Loader::get('Statuses')->getTableName();

            if ($where['status'] === 'all-active') {
                $select->where("$tableName.object_type <> 'thanks' and ($tableName.status_id = 0 or $tableName.status_id in (select id from {$statusT} where is_closed = 0))");
            } else {
                $select->reset(Zend_Db_Select::DISTINCT)
                    ->joinInner(
                        array('s' => $statusT),
                        "$tableName.status_id=s.id AND s.slug = '{$where['status']}'",
                        array()
                    );
            }
        } elseif (isset($where['statuses']) && is_array($where['statuses']) && count($where['statuses'])) {
            $statusT = Singular_Loader::get('Statuses')->getTableName();
            $select->reset(Zend_Db_Select::DISTINCT)
                ->joinInner(
                    array('s' => $statusT),
                    "$tableName.status_id=s.id AND s.id IN (" . implode(',', $where['statuses']) . ")",
                    array()
                );
        }

        if (isset($where['status_id']) && $where['status_id'] != 'all') {
            if ($where['status_id'] === 'all-active') {
                $statusT = Singular_Loader::get('Statuses')->getTableName();
                $select->where("$tableName.object_type <> 'thanks' and ($tableName.status_id = 0 or $tableName.status_id in (select id from {$statusT} where is_closed = 0))");
            } else {
                $select->where("$tableName.status_id = ?", (int)$where['status_id']);
            }
        }

        /* without comments */
        if (!empty($where['without_comments']) && $where['without_comments']) {
            $cTableName = Singular_Loader::get('Resources_Model_DbTable_Comments')
                ->getTableName();
            if (!array_key_exists('c', $select->getPart('from'))) {
                $select->joinLeft(
                    array('c' => $cTableName),
                    "$tableName.id=c.object_id",
                    array()
                );
            }
            $select->where("c.id IS NULL");
        }

        if (isset($order)) {
            if (is_array($order)) {
                if (isset($order["order"]) && $order["order"]) {
                    if (isset($order["sort"]) && $order['sort']) {
                        $sort = "ASC";
                    } else {
                        $sort = "DESC";
                    }
                    switch ($order["order"]) {
                        case 'updated':
                            if (empty($where['text'])) {
                                $select->reset("order");
                            }
                            $select->order("GREATEST($tableName.created_at, IFNULL($tableName.commented_at, '1970-01-01 00:00:00')) DESC");
                            break;
                        default :
                            if (in_array($order["order"], $this->getCols())) {
                                if (empty($where['text'])) {
                                    $select->reset("order");
                                }
                                $select->order("$tableName.{$order["order"]} $sort");
                            }
                            break;
                    }
                }
            } else {
                $select->order($order);
            }
        }

        /* Full-text search */
        if (!empty($where['text'])) {
            if (preg_match('/^tag:([^,]+)$/i', $where['text'], $matches)) {
                $this->prepareTagSearch($matches[1], $select, $tableName);
            } else {
                $merged = clone $select;

                /**
                 * Includes merged objects into search
                 *
                 * @param System_Model_DbTable_Objects $model
                 * @return null
                 */
                $appendMerger = function ($model) use ($select, $merged, $where) {
                    $objectsTableName = $model->getTableName();

                    /** @noinspection PhpParamsInspection */
                    $mergedIds = Singular_Core::_('Db')->getAdapter()->fetchCol(
                        System_Service_FullTextSearch::prepare(
                            $model->select()
                                ->setIntegrityCheck(false)
                                ->from([$objectsTableName])
                                ->reset(Zend_Db_Select::COLUMNS)
                                ->columns(['parent_id'])
                                ->where('state = ?', System_Mapper_State::MERGED),
                            $where['text']
                        )
                    );

                    return is_array($mergedIds) && count($mergedIds)
                        ? $merged->reset(Zend_Db_Select::WHERE)
                            ->reset(Zend_Db_Select::ORDER)
                            ->reset(Zend_Db_Select::GROUP)
                            ->reset(Zend_Db_Select::HAVING)
                            ->where("$objectsTableName.id IN (?)", $mergedIds)
                            ->columns([
                                'relevance'   => 'MAX(0)',
                                'title_rel'   => 'MAX(0)',
                                'content_rel' => 'MAX(0)'
                            ])
                        : null;
                };

                /* Apply Full-text search service */
                $select = System_Service_FullTextSearch::prepare(
                    $select,
                    $where['text'],
                    array(),
                    System_Service_FullTextSearch::MIN_RELEVANCE,
                    System_Service_FullTextSearch::MIN_CONTENT_RELEVANCE,
                    false
                );

                //var_dump($select->assemble()); die();

                /* Apply merged results */
                $mergedTopics = $appendMerger($this);
            }
        }

        /* Search in categories */
        if (isset($where['category_id'])) {
            switch ((string)$where['category_id']) {
                case 'all':
                    break;
                case 'uncat':
                case '0':
                    $select->where("category_id = ?", 0);
                    break;
                default :
                    $select->where("category_id = ?", $where['category_id']);
            }
        }
        if (isset($where['on_moderation']) && $where['on_moderation']) {
            switch ($where['on_moderation']) {
                case 'moderation':
                    $select->where("(`{$this->_name}`.`state` = " . System_Mapper_State::MODERATION . ")");
                    break;
                case 'report-abuse':
                    $select->where(
                        "(`{$this->_name}`.`abuse_or_spam` > 0 AND  `{$this->_name}`.`state` = "
                        . System_Mapper_State::NORMAL . ")"
                    );
                    break;
                default:
                    $select->where(
                        "(`{$this->_name}`.`state` = " . System_Mapper_State::MODERATION
                        . " or (`{$this->_name}`.`abuse_or_spam` > 0 AND  `{$this->_name}`.`state` = "
                        . System_Mapper_State::NORMAL . "))"
                    );
            }
        }

        if (isset($where['id']) && $where['id']) {
            $select->where("`{$this->_name}`.`id` = ?", $where['id']);
        }

        if (isset($where['object_key']) && $where['object_key']) {
            $keyWhere = Singular_Core::_('Db')->getAdapter()->quoteInto(
                "CONCAT(`{$this->_name}`.`id`, DATE_FORMAT(`{$this->_name}`.`created_at`, '-%m-%d')) LIKE ?",
                '%' . $where['object_key'] . '%'
            );

            $propsWhere = Singular_Core::_('Db')->getAdapter()->quoteInto('props.value = ?', $where['object_key']);
            $select->joinLeft(['props' => Singular_Loader::get('PropertiesValues')->getTableName()], "{$this->_name}.id = props.object_id", []);
            $select->where(sprintf('(%s) or (%s)', $keyWhere, $propsWhere));
        }


        /* ASSIGNMENTS FILTERS */
        $oT = $this->getTableName();
        $uT = Singular_Loader::get('Users')->getTableName();

        if (isset($where['date'])) {
            if (!empty($where['created_on_multi_date'])) {
                if ($where['date'] == 'selected-date') {
                    $select->where("$tableName.created_at LIKE ?", $where['created_on_multi_date'] . '%');
                }
                if ($where['date'] == 'before-selected-date') {
                    $select->where("$tableName.created_at < ?", $where['created_on_multi_date']);
                }
                if ($where['date'] == 'after-selected-date') {
                    $select->where("$tableName.created_at > ?", $where['created_on_multi_date']);
                }
                if ($where['date'] == 'selected-date-range') {
                    $select->where("$tableName.created_at > '" . $where['created_on_multi_date'] . "' OR $tableName.created_at LIKE ?",
                        $where['created_on_multi_date'] . '%');
                }
            }
            if (!empty($where['created_on_multi_date_2']) && $where['date'] == 'selected-date-range') {
                $select->where("$tableName.created_at < '" . $where['created_on_multi_date_2'] . "' OR $tableName.created_at LIKE ?",
                    $where['created_on_multi_date_2'] . '%');
            }
        }

        if (isset($where['completed_on'])) {
            if (!empty($where['completed_on_multi_date'])) {
                if ($where['completed_on'] == 'selected-date') {
                    $select->where("$tableName.completed_on LIKE ?", $where['completed_on_multi_date'] . '%');
                }
                if ($where['completed_on'] == 'before-selected-date') {
                    $select->where("$tableName.completed_on < ?", $where['completed_on_multi_date']);
                }
                if ($where['completed_on'] == 'after-selected-date') {
                    $select->where("$tableName.completed_on > ?", $where['completed_on_multi_date']);
                }
                if ($where['completed_on'] == 'selected-date-range') {
                    $select->where("$tableName.completed_on > '" . $where['completed_on_multi_date'] . "' OR $tableName.completed_on LIKE ?",
                        $where['completed_on_multi_date'] . '%');
                }
            }
            if (!empty($where['completed_on_multi_date_2']) && $where['completed_on'] == 'selected-date-range') {
                $select->where("$tableName.completed_on < '" . $where['completed_on_multi_date_2'] . "' OR $tableName.completed_on LIKE ?",
                    $where['completed_on_multi_date_2'] . '%');
            }
        }

        if (isset($where['comments'])) {
            $withoutActivity = isset($where['comments_activity_days']) ? (int)$where['comments_activity_days'] : 0;
            $date = date('Y-m-d H:i:s', strtotime("-{$withoutActivity} day"));
            switch ($where['comments']) {
                case 'all':
                    if ($withoutActivity) {
                        $select->where("{$oT}.commented_at is null and {$oT}.created_at < '{$date}' or {$oT}.commented_at is not null and {$oT}.commented_at < '{$date}'");
                    }

                    break;
                case 'with-comments':
                    if ($withoutActivity) {
                        $select->where("{$oT}.commented_at < ?", $date);
                    }

                    $select->where("{$oT}.commented_at IS NOT NULL");
                    break;
                case 'no-comments':
                    if ($withoutActivity) {
                        $select->where("{$oT}.created_at < ?", $date);
                    }

                    $select->where("{$oT}.commented_at IS NULL");
                    break;
                case 'requires-agent-reply':
                    if ($withoutActivity) {
                        $comTable = Singular_Loader::get('Comments')->getTableName();
                        $agentsIds = Singular_Loader::get('Users')->findAgentsPairs();
                        $agentsIds = array_keys($agentsIds);
                        $select->where("(select count(*) from {$comTable} as act where act.object_id = {$oT}.id and act.created_at > '{$date}' and act.created_by_id in (".implode(',', $agentsIds).")) = 0");
                    }

                    $select->where("{$oT}.awaiting_reply = '1' or {$oT}.commented_at IS NULL");
                    break;
                case 'my-comments':
                    $_selectComment = Singular_Loader::get('Comments')->select(true)
                        ->where("created_by_id = ?", $loggedUser->id)
                        ->where('state = ?', System_Mapper_State::NORMAL)
                        ->where("object_id = {$oT}.id");
                    $select->where('EXISTS (?)', new Zend_Db_Expr($_selectComment));
            }
        }

        if (isset($where['created_by']) && $where['created_by'] != 'all') {
            switch ($where['created_by']) {
                case 'anonymous':
                    $select->where("(SELECT state FROM {$uT} uT WHERE uT.id = {$oT}.created_by_id LIMIT 1) = ?",
                        System_Model_User::STATE_ANONYMOUS);
                    break;
                case 'registered':
                    $select->where("(SELECT state FROM {$uT} uT WHERE uT.id = {$oT}.created_by_id LIMIT 1) = ?",
                        System_Model_User::STATE_NORMAL);
                    break;
                case 'me':
                    $select->where("{$oT}.created_by_id = ?", $loggedUser->id);
                    break;
                case 'selected-filter-user':
                    if (isset($where['created_by_ids']) && is_array($where['created_by_ids'])) {
                        $where['created_by_ids'] = count($where['created_by_ids']) ? $where['created_by_ids'] : array(0);
                        $select->where("{$oT}.created_by_id IN (?)", $where['created_by_ids']);
                    }
                    break;
            }
        }

        if (isset($where['completed_by']) && $where['completed_by'] != 'all') {
            $select->where('completed_on is not null');

            switch ($where['completed_by']) {
                case 'anonymous':
                    $select->where("(SELECT state FROM {$uT} uT WHERE uT.id = {$oT}.completed_by_id LIMIT 1) = ?",
                        System_Model_User::STATE_ANONYMOUS);
                    break;
                case 'registered':
                    $select->where("(SELECT state FROM {$uT} uT WHERE uT.id = {$oT}.completed_by_id LIMIT 1) = ?",
                        System_Model_User::STATE_NORMAL);
                    break;
                case 'me':
                    $select->where("{$oT}.completed_by_id = ?", $loggedUser->id);
                    break;
                case 'selected-filter-user':
                    if (isset($where['completed_by_ids']) && is_array($where['completed_by_ids'])) {
                        $where['completed_by_ids'] = count($where['completed_by_ids']) ? $where['completed_by_ids'] : array(0);
                        $select->where("{$oT}.completed_by_id IN (?)", $where['completed_by_ids']);
                    }
                    break;
            }
        }

        if (isset($where['created_by_role']) && is_array($where['created_by_role'])) {
            $select->where("(SELECT role FROM {$uT} uT WHERE uT.id = {$oT}.created_by_id LIMIT 1) IN (?)",
                $where['created_by_role']);
        }

        if (isset($where['completed_by_role']) && is_array($where['completed_by_role'])) {
            $select->where('completed_on is not null');
            $select->where("(SELECT role FROM {$uT} uT WHERE uT.id = {$oT}.completed_by_id LIMIT 1) IN (?)",
                $where['completed_by_role']);
        }

        Singular_Event::dispatch('afterPrepareSelect', $select);

        if (!empty($where['text']) && isset($mergedTopics) && $mergedTopics instanceof Zend_Db_Select) {
            $this->select()
                ->union(array("({$select->assemble()})", "({$mergedTopics->assemble()})"));
        }

        return $select;
    }

    /**
     * Prepare tag search.
     *
     * @param string|array $searchTags
     * @param Zend_Db_Select $select
     * @param string $tableName
     * @return Zend_Db_Select
     */
    public function prepareTagSearch($searchTags, Zend_Db_Select $select, $tableName)
    {
        if (!is_array($searchTags)) {
            $searchTags = explode(',', $searchTags);
        }

        $conditions = [];

        foreach ($searchTags as $originalTag) {
            $conditions[] = " $tableName.tags = " . Singular_Core::_('Db')->getAdapter()->quote($originalTag);
            $conditions[] = " $tableName.tags LIKE " . Singular_Core::_('Db')->getAdapter()->quote($originalTag . ',%');
            $conditions[] = " $tableName.tags LIKE " . Singular_Core::_('Db')->getAdapter()->quote('%,' . $originalTag);
            $conditions[] = " $tableName.tags LIKE " . Singular_Core::_('Db')->getAdapter()->quote('%,' . $originalTag . ',%');
        }

        $select->where(implode('or', $conditions));

        return $select;
    }

    /**
     * Returns last user created object.
     *
     * @param  int $userId
     * @return Feedback_Model_Topic|null
     */
    public function findLastByUserId($userId)
    {
        $select = $this->getSelect();
        $select->where("{$this->getTableName()}.created_by_id=?", $userId);
        $select->order("{$this->getTableName()}.created_at DESC");
        $select->limit(1);

        return $this->fetchRow($select);
    }

    /**
     * Counts how many responses has posted by given user id
     *
     * @param  int $userId
     *
     * @return int
     */
    public function countByUserId($userId)
    {
        return $this->count(array('created_by_id' => $userId));
    }

    /**
     * Counts responses by given criteria
     *
     * @param  array $where
     * @param  bool  $usePrepareCon
     *
     * @return int
     */
    public function count($where = null, $usePrepareCon = true)
    {
        $select = ($usePrepareCon) ? $this->getSelect() : $this->select(true);

        if (is_array($where)) {
            foreach ($where as $cond => $value) {
                if ($value !== null) {
                    $select->where($cond . '=?', $value);
                }
            }
        } else {
            if (!empty($where) && is_string($where)) {
                $select->where($where);
            }
        }
        $select->reset('columns')
            ->reset('order')
            ->reset('group');

        $select->columns(array('count' => new Zend_Db_Expr("COUNT(DISTINCT({$this->_name}.id))")));

        return $this->getAdapter()->fetchOne($select);
    }

    /**
     * Summ votes of responses by given criteria
     *
     * @param  string|array $where
     *
     * @return int
     */
    public function sumViews($where)
    {
        $select = $this->getSelect();
        $select->reset('where');

        if (!empty($where)) {
            $select->where($where);
        }
        $select->reset('columns')
            ->reset('order')
            ->reset('group');

        $select->columns(array('sum' => new Zend_Db_Expr("SUM(views)")));
        $result = $this->fetchRow($select);

        return $result['sum'];
    }

    /**
     * Count all comments by user
     *
     * @param  System_Model_User $user
     *
     * @return int
     */
    public function countAllComment(System_Model_User $user)
    {
        $commentsTable = Singular_Loader::get('Comments')->getTableName();
        $objectsTable = Singular_Loader::get('Objects')->getTableName();

        $select = Singular_Core::_('Db')->getAdapter()->select()
            ->from(['comments' => $commentsTable])
            ->reset(Zend_Db_Select::COLUMNS)
            ->columns(['COUNT(comments.id)'])
            ->joinLeft(['objects' => $objectsTable], "comments.object_id = objects.id", [])
            ->where('comments.state = ?', System_Mapper_State::NORMAL)
            ->where('objects.state IN (?)', [System_Mapper_State::NORMAL, System_Mapper_State::ARCHIVED])
            ->where('objects.is_private <= ?', (int)$user->role()->isAgent())
            ->where('comments.is_private = 0');

        if (!$user->role()->isAgent()) {
            $select->where('objects.module != "helpdesk" OR objects.created_by_id = ?', ($user->isGuest() ? 0 : $user->id));
        }

        return Singular_Core::_('Db')->getAdapter()->fetchOne($select);
    }

    /**
     * Return array with category_id => sum objects
     *
     * @param  array|System_Model_Category $cat
     * @param  string                      $private
     * @param  string                      $module
     *
     * @return array
     */
    public function sumByCategoriesIds($cat, $private = null, $module)
    {
        $sumSel = $this->getSelect()
            ->where("{$this->_name}.module = ?", $module)
            ->where("{$this->_name}.is_private <= ?")
            ->reset(Zend_Db_Select::COLUMNS)
            ->columns(array("category_id", "sum" => new Zend_Db_Expr("COUNT({$this->_name}.id)")))
            ->group("category_id");

        /* Fetch object with active types only */
        $objectTypesRowset = Singular_Loader::get('ObjectTypes')->findByModule(
            array($module)
        );
        if (!empty($objectTypesRowset)) {
            $types = array();
            foreach ($objectTypesRowset as $typeRow) {
                if ((bool)$typeRow->is_active) {
                    $types[] = $typeRow->slug;
                }
            }
            if (!empty($types)) {
                $sumSel->where("object_type IN (?)", $types);
            }
        }

        return $this->getAdapter()->fetchPairs($sumSel, array((int)$private));
    }

    /**
     * Returns topic by slug
     *
     * @param  string            $slug
     * @param  System_Model_User $user
     *
     * @throws Zend_Exception
     * @throws Zend_Acl_Exception
     * @return System_Model_object
     */
    public function findBySlug($slug, System_Model_User $user)
    {
        $select = $this->select(true)->where('slug = ?', $slug);
        $row    = $this->fetchRow($select);
        if ($row instanceof System_Model_Object) {
            $row = $row->convertObject();
            if (!$row->canSee($user)) {
                if ($user->isGuest()) {
                    Singular_Core::_('Environment')->gotoLoginAndBackHere();
                } else {
                    Singular_Core::_('Environment')->accessDenied();
                }
            } else {
                return $row;
            }
        } else {
            Singular_Core::_('Environment')->notFound();
        }
    }

    /**
     * Returns object by id
     *
     * @param  int               $id
     * @param  System_Model_User $user
     *
     * @return System_Model_Object
     */
    public function findById($id, System_Model_User $user)
    {
        $select = $this->select();
        $select->where('id=?', $id);
        $object = $this->fetchRow($select);
        if ($object instanceof System_Model_Object) {
            $object = $object->convertObject();
            $object = $object->canSee($user) ? $object : null;
        }

        return $object;
    }

    /**
     * Returns object by id
     *
     * @param  array             $ids
     * @param  System_Model_User $user
     *
     * @return System_Model_Object[]|null
     */
    public function findByIds($ids, System_Model_User $user)
    {
        $select = $this->select();
        $select->where('id IN (?)', $ids);
        $objects = $this->fetchAll($select);
        if (!count($objects)) {
            return null;
        }
        $result = null;
        foreach ($objects as $_o) {
            $obj = $_o->convertObject();
            if ($obj->canSee($user)) {
                $result[] = $obj;
            }
        }

        return $result;
    }

    /**
     * Count all topics by type
     *
     * @param  System_Model_User $user
     * @param bool               $countPrivate
     * @param bool               $countFeatured
     * @param int                $countState
     *
     * @return array|null
     */
    public function countByObjectType(System_Model_User $user, $countPrivate = true, $countFeatured = false, $countState = 0)
    {
        $objectTypes = Singular_Loader::get('ObjectTypes')->findActive($this->createRow()->getModule());
        $typeCon     = array();

        if (!count($objectTypes)) {
            return null;
        }

        foreach ($objectTypes as $type) {
            $typeCon[] = "COUNT(CASE WHEN object_type = '{$type->slug}' THEN 1 END) AS {$type->slug}";
        }
        $sel = $this->select(true)->reset(Zend_Db_Select::COLUMNS)->columns($typeCon);
        if ($countPrivate) {
            $sel->where("{$this->_name}.is_private IN (?)", (bool)$user->role()->isAgent() ? array(0, 1) : 0);
        } else {
            $sel->where("{$this->_name}.is_private = 0");
        }

        if ($countFeatured) {
            $sel->where("{$this->_name}.featured <> ?", 0);
        }

        if ($countState !== 0) {
            $sel->where("{$this->_name}.state IN (?)", [$countState]);
        } else {
            $sel->where("{$this->_name}.state IN (?)", [System_Mapper_State::NORMAL, System_Mapper_State::ARCHIVED]);
        }

        $result = $this->fetchRow($sel);

        return $result ? $result->toArray() : array();
    }

    /**
     * Export Excel
     *
     * @param $objects
     * @return string
     */
    public function exportExcel($objects)
    {
        if (!is_array($objects) && !($objects instanceof System_Model_Rowset)) {
            return null;
        }

        $view = Zend_Layout::getMvcInstance()->getView();
        $view->assign('objects', $objects);

        return $view->render('_export_objects_excel.phtml');
    }

    /**
     * Export Excel
     *
     * @param $objects
     * @return string
     */
    public function exportCSV($objects)
    {
        if (!is_array($objects) && !($objects instanceof System_Model_Rowset)) {
            return null;
        }

        $view = Zend_Layout::getMvcInstance()->getView();
        $topicFields = Singular_Loader::get('Properties')->findByAlias('feedback_topic');
        $ticketFields = Singular_Loader::get('Properties')->findByAlias('helpdesk_ticket');

        $row = array(
            $view->t('Title'),
            $view->t('Description'),
            $view->t('Object Type'),
            $view->t('Status'),
            $view->t('Category'),
            $view->t('Visibility'),
            $view->t('Responsible'),
            $view->t('Created By'),
            $view->t('Completed By'),
            $view->t('Completed On'),
            $view->t('Age'),
            $view->t('Number of Comments'),
            $view->t('Latest Activity'),
            $view->t('Source'),
            $view->t('Votes')
        );

        foreach ($topicFields as $field) {
            $row[] = $view->escape($field->label);
        }

        foreach ($ticketFields as $field) {
            $row[] = $view->escape($field->label);
        }

        $structure = array($row);

        foreach ($objects as $object) {
            if ($object instanceof System_Model_Object) {
                $row = array(
                    $object->title,
                    $object->content,
                    $view->t(Singular_Loader::get('ObjectTypes')->findBySlug($object->object_type)->title),
                    $view->t($object->getStatus() ? $object->getStatus()->title : ' - '),
                    $object->getAssignmentValues('category', true),
                    $object->getAssignmentValues('visibility', true),
                    $object->getAssignmentValues('responsible', true),
                    $object->getAssignmentValues('created_by', true),
                    $object->getAssignmentValues('completed_by', true),
                    $object->getAssignmentValues('completed_on', true),
                    $object->getAssignmentValues('age', true),
                    $object->getAssignmentValues('comments', true),
                    $object->getAssignmentValues('activity', true),
                    $object->getAssignmentValues('source', true),
                    $object->getAssignmentValues('votes_count', true),
                );

                foreach ($topicFields as $field) {
                    $row[] = $field->module == $object->module ? $view->escape($field->findTextValue($object)) : '';
                }

                foreach ($ticketFields as $field) {
                    $row[] = $field->module == $object->module ? $view->escape($field->findTextValue($object)) : '';
                }

                $structure[] = $row;
            }
        }

        $arrayToCSV = function ($array) {
            $arrayToCSVRow = function ($value_set) {
                $values = array();
                foreach ($value_set as $value) {
                    $value = str_replace('"', '""', $value);
                    if (strpos($value, ',') !== false || strpos($value, '"') !== false || strpos($value,
                            "\n") !== false || strpos($value, "\r") !== false
                    ) {
                        $values[] = '"' . $value . '"';
                    } else {
                        $values[] = $value;
                    }
                }

                return implode(',', $values) . "\n";
            };
            if (is_array($array)) {
                $result = '';
                foreach ($array as $value_set) {
                    $result .= $arrayToCSVRow($value_set);
                }

                return $result;
            } else {
                return null;
            }
        };

        return $arrayToCSV($structure);
    }

    /**
     * Get Assignments
     *
     * @param array             $options
     * @param System_Model_User $user
     * @return array|System_Model_Object[]|int
     */
    public function getAssignments($options, System_Model_User $user)
    {
        if (empty($options['object_type'])) {
            $types                    = Singular_Loader::get('ObjectTypes')->findForElementSelect(array(
                'feedback',
                'helpdesk'
            ));
            $options['object_type']   = array_values($types);
            $options['object_type'][] = 'ticket';
        }

        if (empty($options['order'])) {
            $options['order']  = 'new_updated';
        }

        $options['active'] = true;
        $select            = parent::getSelect();
        if (empty($options['show_archived'])) {
            $state             = empty($options['archived']) ? System_Mapper_State::DEFAULT_STATE : System_Mapper_State::ARCHIVED;
            $select->where("{$this->getTableName()}.state=?", $state);
        } else {
            $select->where("{$this->getTableName()}.state in (?)", [System_Mapper_State::DEFAULT_STATE, System_Mapper_State::ARCHIVED]);
        }
        $select->distinct(true);

        if ($user->role()->denied('can_manage_tickets')) {
            $select->where("{$this->getTableName()}.`module` != 'helpdesk'");
        }

        if (!empty($options['limit'])) {
            $select->limit($options['limit']);
            unset($options['limit']);
        }

        $mT = Singular_Loader::get('Modules')->getTableName();
        $select->join(array('mt' => $mT), "mt.`name` = {$this->getTableName()}.`module` AND mt.`active` > 0", array());

        if (array_key_exists('sources', $options)) {
            if (is_array($options['sources']) && count($options['sources'])) {
                $select->where("{$this->getTableName()}.source IN (?)", $options['sources']);
            }

            unset($options['sources']);
        }

        if (!empty($options['properties']) && is_array($options['properties']) && count($options['properties'])) {
            $cfvDb = Singular_Loader::get('PropertiesValues');
            $adapter = Singular_Core::_('Db')->getAdapter();

            $topicConditions = [];
            $ticketConditions = [];

            $propertiesMap = $adapter->fetchPairs(
                $adapter->select()
                    ->from(Singular_Loader::get('Properties')->getTableName())
                    ->reset(Zend_Db_Select::COLUMNS)
                    ->columns(['id', 'module'])
                    ->where('module in (?)', ['feedback', 'helpdesk'])
                    ->where('field_type <> "option"')
            );

            foreach ($options['properties'] as $field) {
                $field = (object)$field;
                $cfvSelect = $cfvDb->select()
                    ->setIntegrityCheck(false)
                    ->from(array('cfv' => $cfvDb->getTableName()))
                    ->reset(Zend_Db_Select::COLUMNS)
                    ->columns(array('COUNT(*)'))
                    ->where("cfv.object_id = {$this->getTableName()}.id")
                    ->where('cfv.field_id = ?', $field->id);

                $anyValues = null;

                if (is_array($field->value) && count($field->value)) {
                    $condition = "cfv.value IN (?)";

                    foreach ($field->value as $value) {
                        $value = (int)$value;

                        if ($value) {
                            $condition .= " OR cfv.value LIKE '{$value}|%' OR cfv.value LIKE '%|{$value}' OR cfv.value LIKE '%|{$value}|%'";
                        } else {
                            $anyValues = clone $cfvSelect;
                            $condition .= " OR cfv.value is null or cfv.value = ''";
                        }
                    }

                    $cfvSelect->where($condition, $field->value);
                } else {
                    preg_match('/date\|([^\|]*)\|([^\|]*)/', $field->value, $matches);

                    if (count($matches)) {
                        if (!empty($matches[1])) {
                            $cfvSelect->where('cfv.value >= ?', Resources_Service_Properties::dateConverter($matches[1]));
                        }

                        if (!empty($matches[2])) {
                            $cfvSelect->where('cfv.value < ?', Resources_Service_Properties::dateConverter($matches[2]));
                        }
                    }
                }

                if (!empty($propertiesMap[$field->id]) && $propertiesMap[$field->id] === 'helpdesk') {
                    $ticketConditions[] = "(({$cfvSelect->assemble()}) > 0" . ($anyValues ? " or ({$anyValues->assemble()}) = 0" : "") . ')';
                } else {
                    $topicConditions[] = "(({$cfvSelect->assemble()}) > 0" . ($anyValues ? " or ({$anyValues->assemble()}) = 0" : "") . ')';
                }

                //$select->where("({$cfvSelect->assemble()}) > 0" . ($anyValues ? " or ({$anyValues->assemble()}) = 0" : ""));
            }

            $propConditions = [];

            if (count($ticketConditions)) {
                $propConditions[] = implode(' and ', $ticketConditions);
            }

            if (count($topicConditions)) {
                $propConditions[] = implode(' and ', $topicConditions);
            }

            if (count($propConditions)) {
                $select->where(implode(' or ', $propConditions));
            }
        }

        if (!empty($options['tags'])) {
            $this->prepareTagSearch($options['tags'], $select, $this->getTableName());
            unset($options['tags']);
        }

        $this->prepareSelect($select, $options, null, $user);

        return $this->fetchAll($select);
    }

    /**
     * Group Assignments
     *
     * @param Zend_Db_Table_Rowset_Abstract|System_Model_Object[] $objects
     * @param string|bool                                         $field
     * @return array
     */
    public function groupAssignments($objects, $field = false)
    {
        $result = array();

        /**
         * @param System_Model_Object $object
         * @param string              $field
         * @return string
         */
        $grouping = function ($object, $field) {
            $view = Zend_Layout::getMvcInstance()->getView();

            switch ($field) {
                case 'category_id':
                    return $view->t(
                        Singular_Loader::get('ObjectTypes')->findBySlug($object->object_type)->title
                    ) . ': ' . $object->getCategory(true)->name;
                case 'object_type':
                    return Singular_Loader::get('ObjectTypes')->findBySlug($object->object_type)->plural_title;
                case 'status_id':
                    return $object->status_id
                        ? $view->t(
                            Singular_Loader::get('ObjectTypes')->findBySlug($object->object_type)->title
                        ) . ': ' . $view->t(Singular_Loader::get('Statuses')->findById($object->status_id))
                        : $view->t('No Status');
                case 'responsible_id':
                    return $object->getResponsible() ? $object->getResponsible()->getName(false) : $view->t('No Responsible');
                case 'is_private':
                    return $object->is_private ? $view->t('Private') : $view->t('Normal');
                case 'archived':
                    return $view->t('Archived');
                case 'source':
                    $sources = System_Mapper_Source::getValueNamePairs();
                    return array_key_exists($object->source, $sources) ? $sources[$object->source] : 'No Source';
                default:
                    return $field;
            }
        };

        $i = 0;
        foreach ($objects as $object) {
            /** @var System_Model_Object $object */
            $returnStatus = false;
            if ($field == 'status_id' && $object->status_id == 0) {
                $returnStatus = true;
                $defStatus    = Singular_Loader::get('Statuses')->findDefaultByObjectType($object->object_type,
                    $object->module);
                if ($defStatus instanceof System_Model_Status) {
                    $object->status_id = $defStatus->id;
                }
            }

            if ($field === 'responsible_id' && !$object->getResponsible()) {
                $key = 0;
            } else {
                $key = $field ? ($field === 'archived' ? $field : $object->$field) : 'not_grouped';
            }

            $result[$key]['name'] = $grouping($object, $field);
            if ($returnStatus) {
                $object->status_id = 0;
            }
            $result[$key]['items'][] = $object->convertObject();

            if (++$i >= 100) {
                break;
            }
        }

        return $result;
    }
}
