<?php
/**************************************************************
 * Copyright USWebStyle Inc., 2011-2012. All Rights Reserved.
 **************************************************************
 * NOTICE:  This source code is the property of USWebStyle, Inc.
 * In no case shall you rent, lease, lend, redistribute
 * any of below source code to a 3rd party individual or entity.
 *
 * @category   System
 * @package    System_Model
 * @copyright  USWebStyle Inc., 2011-2012 (http://www.uswebstyle.com)
 * @license    http:/www.useresponse.com/licensing (Commercial)
 */

/**
 * users row class
 *
 * @uses       System_Model_Row
 * @category   System
 * @package    System_Model
 */
class System_Model_User extends System_Model_Row
{
    /**
     * Name of the class of the Zend_Db_Table_Abstract object.
     *
     * @var string
     */
    protected $_tableClass = 'System_Model_DbTable_Users';

    /**
     * Magic method __tostring
     *
     * @return string
     */
    public function  __toString()
    {
        return $this->getName();
    }

    /**
     * Allows pre-update logic to be applied to row.
     * Subclasses may override this method.
     *
     * @return void
     */
    protected function _update()
    {
        if (isset($this->_modifiedFields['photo']) && $this->_modifiedFields['photo']) {
            if (empty($this->photo)) {
                $this->_modifiedFields['photo'] = false;
                $this->photo = $this->_cleanData['photo'];
            } elseif (!empty($this->_cleanData['photo'])) {
                @unlink(ROOT_PATH . '/public/files/avatars/' . $this->_cleanData['photo']);
            }
        }
    }

    /**
     * Allows post-insert logic to be applied to row.
     * Subclasses may override this method.
     *
     * @see    parent::_postInsert()
     * @return void
     */
    protected function _postInsert()
    {
        $this->_cleanDefaultCache();
    }
    
    /**
     * Allows post-update logic to be applied to row.
     * Subclasses may override this method.
     *
     * @return void
     */
    protected function _postUpdate()
    {
        $this->_cleanDefaultCache();
    }
    
    /**
     * Allows post-delete logic to be applied to row.
     * Subclasses may override this method.
     *
     * @see    parent::_postDelete()
     * @return void
     */
    protected function _postDelete()
    {
        $this->_cleanDefaultCache();
    }
    
    /**
     * Clean default cache
     *
     * @return void
     */
    protected function _cleanDefaultCache()
    {
        $cacheTags = array('responses', 'comments');
        $cache = Singular_Runtime::extract('cachemanager')->getCache('default');
        $cache->clean(Zend_Cache::CLEANING_MODE_MATCHING_ANY_TAG, $cacheTags);
    }
    
    /**
     * Returns user name
     *
     * @return string
     */
    public function getName($escape = true)
    {
        
        if (!empty($this->full_name)) {
            if ($escape) {
                $view = Zend_Layout::getMvcInstance()->getView();
                $name = $view->escape($this->full_name);
            } else {
                $name = $this->full_name;
            }
            return $name;
        } elseif (!empty($this->email)) {
            return $this->email;
        } else {
            return '';
        }
    }

    /**
     * Returns avatar source path
     *
     * @return string
     */
    public function getAvatarSrc()
    {
        $view = Zend_Layout::getMvcInstance()->getView();
        if ($this->photo) {
            $src = $view->baseUrl(AVATAR_URI . '/' . $this->photo);
        } else {
            $src = $view->baseUrl(AVATAR_URI . '/avatar-default.png');
        }
        return $src;
    }

    /**
     * Returns user role
     *
     * @return System_Model_Role
     */
    public function getRole()
    {
        $rolesDb = Singular_Loader::get('System_Model_DbTable_Roles');
        return $rolesDb->findByRole($this->role);
    }

    /**
     * Returns ISO-formatted `created_at` field
     *
     * @return Zend_Date
     */
    public function getCreatedAt()
    {
        return new Zend_Date($this->created_at, Zend_Date::ISO_8601);
    }

    /**
     * Returns user notofications
     *
     * @return Zend_Db_Table_Rowset
     */
    public function getNotifications()
    {
        if (!$this->getProperty('notifications')) {
            $notificationsDb = Singular_Loader::get('System_Model_DbTable_Notification');
            $this->setProperty('notifications', $notificationsDb->findByUserId($this->id));
        }
        return $this->getProperty('notifications');
    }

    /**
     * Deletes user avatar
     *
     * @return void
     */
    public function deletePhoto()
    {
        @unlink(AVATAR_PATH . DS . $this->photo);
        $this->photo = "";
        $this->getTable()->update(array('photo' => ''), "id={$this->id}");
    }

    /**
     * Sets user password
     *
     * @see    Singular_Core_Auth::makePassword() password hasher
     * @param  string $password
     * @return void
     */
    public function setPassword($password)
    {
        $this->password = Singular_Core::_('Auth')->makePassword($password);
    }

    /**
     * Returns user setting by key
     *
     * @param  string $key
     * @return string
     */
    public function getConfig($key)
    {
        $usersConfigDb = Singular_Loader::get('System_Model_DbTable_UsersConfigs');
        $config = $usersConfigDb->findByKeyAndUserId($key, $this->id);
        if ($config instanceof  System_Model_UserConfig) {
            return $config->value;
        } else {
            return Singular_Core::_('Settings')->getSetting($key);
        }
    }

    /**
     * Returns user settings
     *
     * @return array
     */
    public function getConfigValues()
    {
        if (!isset($this->_properties['config_values'])) {
            $usersConfigDb = Singular_Loader::get('System_Model_DbTable_UsersConfigs');
            $configs = $usersConfigDb->findValuesByUserId($this->id);
            
            $defaultSettings = Singular_Core::_("settings")->getModuleSettings('system', 'user')->toArray();
            foreach ($defaultSettings as $key=>$val) {
                if (!isset($configs[$key])) {
                    $configs[$key] = $val['value'];
                }
            }
            $this->_properties['config_values'] = $configs;
        }
        return $this->_properties['config_values'];
    }

    /**
     * Returns user last activity
     *
     * @return Zend_Date
     */
    public function getLastActivity()
    {
        $responsesDb = Singular_Loader::get("System_Model_DbTable_Responses");
        $response = $responsesDb->findLastByUserId($this->id);

        $commentsDb = Singular_Loader::get("Resources_Model_DbTable_ResponsesComments");
        $comment = $commentsDb->findLastByUserId($this->id);

        if ($response instanceof System_Model_Response
                && $comment instanceof Resources_Model_ResponseComment) {
            if ($comment->getCreatedAt()
                    ->getTimestamp() > $response->getCreatedAt()->getTimestamp()) {
                return $comment->getCreatedAt();
            } else {
                return $response->getCreatedAt();
            }
        } elseif ($comment instanceof Resources_Model_ResponseComment) {
            return $comment->getCreatedAt();
        } elseif ($response instanceof System_Model_Response) {
            return $response->getCreatedAt();
        }
        return $this->getCreatedAt();
    }

    /**
     * Checks when current user is admin
     *
     * @return bool
     */
    public function isAdmin()
    {
        if ($this->isAllowed('can_access_admin')) {
            return true;
        }
        return false;
    }

    /**
     * Checks when current user is employee
     *
     * @return bool
     */
    public function isEmployee()
    {
        if ($this->isAllowed('can_access_admin')
                || $this->isAllowed('representative')) {
            return true;
        }
        return false;
    }

    /**
     * Checks when current user is guest
     *
     * @return bool
     */
    public function isGuest()
    {
        if ($this->role == 'guest') {
            return true;
        }
        return false;
    }

    /**
     * Checks when current user can see response
     *
     * @param  System_Model_Response $response
     * @return bool
     */
    public function canSeeResponse($response)
    {
        if ($this->isAdmin()) {
            return true;
        }
        $result = true;
        if($response->deleted_by_id && !$this->canManageResponses()) {
            $result = false;
        }
        $params = array(
            'result' => $result,
            'response' => $response
        );
        Singular_Event::dispatch('canSeeResponse', $params);
        return $params['result'];
    }
    
    /**
     * Checks when current user can see comment
     *
     * @param  Resources_Model_ResponseComment $comment
     * @return bool
     */
    public function canSeeComment($comment)
    {
        if ($this->isAdmin()) {
            return true;
        }
        $result = true;
        if($comment->deleted_by_id && !$this->canManageResponses()) {
            $result = false;
        }
        $params = array(
            'result' => $result,
            'comment' => $comment
        );
        Singular_Event::dispatch('canSeeComment', $params);
        return $params['result'];
    }
    
    /**
     * Checks when current user can manage people
     *
     * @param  string|null $user
     * @return bool
     */
    public function canManagePeople($user = null)
    {
        if ($this->isAllowed('manage_people')
                || $this->isAllowed('can_access_admin')) {
            if ($user instanceof System_Model_User) {
                if ($user->id == $this->id) {
                    return true;
                }
                if ($this->isAdmin() && $user->isAdmin()) {
                    return true;
                }
                if ($this->isAllowed('manage_people')
                        && $user->isAdmin()) {
                    return false;
                }
            }
            return true;
        }
        return false;
    }

    /**
     * Checks when current user can use activity log
     *
     * @return bool
     */
    public function canUseActivityLog()
    {
        if ($this->isAllowed('can_use_active_log') || $this->isAllowed('can_access_admin')) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Checks when current user can manahe responses
     *
     * @return bool
     */
    public function canManageResponses()
    {
        if ($this->isAllowed('manage_responses') || $this->isAllowed('can_access_admin')) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Checks when current user can add comment
     *
     * @return bool
     */
    public function canAddComment()
    {
        if (!$this->isGuest()) {
            return true;
        }
        return false;
    }

    /**
     * Checks when current user can add vote
     *
     * @return bool
     */
    public function canAddVote()
    {
        if (!$this->isGuest()) {
            return true;
        }
        if (Singular_Core::_("Settings")->getSetting("anonymous_votes")) {
            return true;
        }
        return false;
    }

    /**
     * Checks when specified resource is allowed for current user
     *
     * @param  string $resource
     * @return bool
     */
    public function isAllowed($resource)
    {
        $acl = Singular_Runtime::extract('acl');
        if ($acl->isAllowed($this->role, Singular_Acl::RESOURCE_TYPE_OBJECT . Singular_Acl::TYPE_DIVIDER . $resource)) {
            return true;
        }
        return false;
    }

    /**
     * Saves the properties to the database.
     *
     * This performs an intelligent insert/update, and reloads the
     * properties with fresh data from the table on success.
     *
     * @see    parent::save()
     * @param  array|null $data
     * @return mixed The primary key value(s), as an associative array if the
     *     key is compound, or a scalar if the key is single-column.
     */
    public function save($data = null)
    {
        $result = parent::save($data);
        if ($result) {
            if (is_array($data['config'])) {
                $this->saveConfig($data['config']);
            }
        }
        return $result;
    }

    /**
     * Updates user settings
     *
     * @param  array $data
     * @throws Zend_Exception
     * @return void
     */
    public function saveConfig($data)
    {
        if (!is_array($data)) {
            throw new Zend_Exception('Invalid Params \'data\'');
        }

        $usersConfigsDb = Singular_Loader::get('System_Model_DbTable_UsersConfigs');

        foreach ($data as $key=>$value) {
            if ($value != '') {
                $config = $usersConfigsDb->findByKeyAndUserId($key, $this->id);
                if ($config instanceof System_Model_UserConfig) {
                    if ($config->value != $value) {
                        $config->value = $value;
                        $config->save();
                    }
                } else {
                    $config = new System_Model_UserConfig();
                    $config->config_key = $key;
                    $config->user_id = $this->id;
                    $config->value = $value;
                    $config->save();
                }
            }
        }
    }
    
    /**
     * Delete user.
     *
     * @see    parent::delete()
     * @param  null
     * @return void
     */
    public function delete()
    {
        $photo = $this->photo;
        $result = parent::delete();
        if ($result) {
            @unlink(AVATAR_PATH . DS . $photo);
        }
    }
}
