<?php
/**************************************************************
 * Copyright UseResponse Inc., 2011-2014. All Rights Reserved.
 **************************************************************
 * NOTICE:  This source code is the property of UseResponse Inc.
 * In no case shall you rent, lease, lend, redistribute
 * any of below source code to a 3rd party individual or entity.
 *
 * @category   Resources
 * @package    Resources_Service
 * @copyright  UseResponse Inc., 2011-2014 (https://www.useresponse.com)
 * @license    https://www.useresponse.com/eula_license-agreement (Commercial)
 */

use Useresponse\Application\Paths;
use Useresponse\Filesystem\Storage;

/**
 * Files uploader services
 *
 * @category   Resources
 * @package    Resources_Service
 */
class Resources_Service_Uploader
{

    /**
     * Available extensions.
     *
     * @var array
     */
    protected static $availableExtensions = [
        'jpg', 'jpeg', 'gif', 'png', 'bmp', 'ico', 'pdf', 'doc',
        'rtf', 'txt', 'zip', 'rar', 'docx', 'gz', 'tar', 'xls',
        'xlsx', 'ppt', 'pptx', 'pps', 'ppsx', 'csv', 'jar', 'log',
        'xml', 'bin', 'mp3', 'mp4', 'md', 'webm', 'swf'
    ];

    /**
     * Returns all available extensions.
     *
     * @return array
     */
    public static function availableExtensions()
    {
        return self::$availableExtensions;
    }

    /**
     * Returns true if specified file uploaded.
     *
     * @param string $filename
     * @return bool
     */
    public static function isUploaded($filename)
    {
        return !empty($_FILES[$filename]['name']);
    }

    /**
     * Returns true if file valid.
     * 
     * @param string $filename
     * @return bool
     */
    public static function isUploadFileValid($filename)
    {
        $result = false;

        if (self::isUploaded($filename)) {
            $extension = pathinfo($_FILES[$filename]['name'], PATHINFO_EXTENSION);

            $result = $extension && in_array(strtolower($extension), self::$availableExtensions, true);
        }

        return $result;
    }

    /**
     * Move uploaded file from php temp dir to /public/:specified
     *
     * @param string $filename
     * @param string $destination
     * @param string $newFileName
     * @return string
     */
    public static function moveUploadedTo($filename, $destination, $newFileName = null)
    {
        $file = fopen($_FILES[$filename]['tmp_name'], 'r');

        if (!is_resource($file)) {
            throw new LogicException('File was not uploaded');
        }

        $extension = pathinfo($_FILES[$filename]['name'], PATHINFO_EXTENSION);
        $newName = null !== $newFileName ? $newFileName : md5(microtime()) . '.' . $extension;
        $destination .= ('/' . $newName);

        if (Storage::disk('public')->exists($destination)) {
            Storage::disk('public')->delete($destination);
        }

        Storage::disk('public')->writeStream($destination, $file);

        return $newName;
    }

    /**
     * Is file extension allowed
     *
     * @param string $filename
     * @param array $extensions
     * @return bool
     */
    public static function isExtensionAllowed($filename, array $extensions = null)
    {
        $extension = pathinfo($filename, PATHINFO_EXTENSION);
        $extensions = $extensions ?: self::$availableExtensions;

        return in_array(strtolower($extension), $extensions, true);
    }

    /**
     * Treat Inline Images.
     *
     * @param string $text
     * @param bool $asBbCode
     * @param bool $relative
     * @return string
     */
    public static function treatInlineImages($text, $asBbCode = true, $relative = true)
    {
        $regexp = $asBbCode
            ? '/\[img\]public:\/\/(.*)\[\/img\]/is'
            : '/<img(.*?)src=[\'"]public:\/\/(.*)[\'"](.*?)>/is';

        $replacement = $asBbCode
            ? '[img]'.publicFile('$1', $relative).'[/img]'
            : '<img$1src="'.publicFile('$2', $relative).'"$3>';

        while (preg_match($regexp, $text)) {
            $text = preg_replace($regexp, $replacement, $text);
        }

        return $text;
    }

    /**
     * Get image size.
     *
     * @param string $path
     * @return array
     */
    public static function getImageSize($path)
    {
        $result = null;
        $temporaryCopy = null;

        if (preg_match('/^'.preg_quote(publicFile(null), '/').'\/(.*)$/is', $path, $matches)) {
            $temporaryCopy = 'public://' . $matches[1];
        } elseif (preg_match('/^'.preg_quote(publicFile(null, false), '/').'\/(.*)$/is', $path, $matches)) {
            $temporaryCopy = 'public://' . $matches[1];
        } elseif (preg_match('/^'.preg_quote(asset(null), '/').'\/(.*)$/is', $path, $matches)) {
            $temporaryCopy = 'app://' . $matches[1];
        } elseif (preg_match('/^'.preg_quote(asset(null, false), '/').'\/(.*)$/is', $path, $matches)) {
            $temporaryCopy = 'app://' . $matches[1];
        } elseif (preg_match('#^(app|public|local)\:\/\/#', $path)) {
            $temporaryCopy = $path;
        }

        if (null !== $temporaryCopy) {
            if (!Storage::exists($temporaryCopy)) {
                return null;
            }

            $tmpFilename = rand(100, 1000) . '_' . time() . '.tmp';
            Storage::copyBetween($temporaryCopy, 'local://work/' . $tmpFilename);
            $result = getimagesize(Paths::baseRoot('work/' . $tmpFilename));
            Storage::delete('local://work/' . $tmpFilename);
        }

        if (null === $result) {
            try {
                $size = getimagesize($path);
                $result = $size;
            } catch (Exception $e) {}
        }

        return $result;
    }

    /**
     * Returns max upload size.
     *
     * @return float|int
     */
    public static function maxUploadSize()
    {
        $maxSize = -1;

        if ($maxSize < 0) {
            $maxSize = self::parseSize(ini_get('post_max_size'));
            $uploadMax = self::parseSize(ini_get('upload_max_filesize'));

            if ($uploadMax > 0 && $uploadMax < $maxSize) {
                $maxSize = $uploadMax;
            }
        }

        return $maxSize;
    }

    /**
     * Parse size.
     *
     * @param $size
     * @return float
     */
    protected static function parseSize($size) {
        $unit = preg_replace('/[^bkmgtpezy]/i', '', $size);
        $size = preg_replace('/[^0-9\.]/', '', $size);

        if ($unit) {
            return round($size * pow(1024, stripos('bkmgtpezy', $unit[0])));
        } else {
            return round($size);
        }
    }
}