<?php
/**************************************************************
 * Copyright USWebStyle Inc., 2011-2012. All Rights Reserved.
 **************************************************************
 * NOTICE:  This source code is the property of USWebStyle, Inc.
 * In no case shall you rent, lease, lend, redistribute
 * any of below source code to a 3rd party individual or entity.
 *
 * @category   Updates
 * @package    Updates_Handler
 * @copyright  USWebStyle Inc., 2011-2012 (http://www.uswebstyle.com)
 * @license    http:/www.useresponse.com/licensing (Commercial)
 */

/**
 * Knowledge Base module handler
 *
 * @uses       Singular_Event_Listener_Abstract
 * @category   Updates
 * @package    Updates_Handler
 */
class KnowledgeBase_Handler_Events extends Singular_Event_Listener_Abstract
{
    /**
     * Binds aggregates on specified events.
     *
     * @return void
     */
    public static function bindListeners ()
    {
        self::attachAggregate('postDispatchOnce',       array(__CLASS__, 'onPostDispatchOnce'));
        self::attachAggregate('buildNavigation',        array(__CLASS__, 'onNavigation'));
        self::attachAggregate('formInitAfter',          array(__CLASS__, 'onFormInitAfter'));
        self::attachAggregate('afterSave',              array(__CLASS__, 'onAfterSave'));
        self::attachAggregate('viewVarSet',             array(__CLASS__, 'onViewVarSet'));
        self::attachAggregate('select',                 array(__CLASS__, 'onSelect'));
        self::attachAggregate('componentRenderAfter',   array(__CLASS__, 'onComponentRenderAfter'));
        self::attachAggregate('routeShutdown',          array(__CLASS__, 'onRouteShutdown'));
        self::attachAggregate('onMainPage',             array(__CLASS__, 'onMainPage'));
        self::attachAggregate('moduleDeactivateBefore', array(__CLASS__, 'onModuleDeactivateBefore'));
        self::attachAggregate('moduleUninstallBefore',  array(__CLASS__, 'onModuleUninstallBefore'));
        self::attachAggregate('widgetStack',            array(__CLASS__, 'onWidgetStack'));
    }

    /**
     * Subscribes to "navigation" event.
     *
     * Builds navigation
     *
     * @param  Singular_Navigation $navigation
     * @return void
     */
    public static function onNavigation ($navigation)
    {
        $loggedUser = Singular_Runtime::extract("loggedUser");

        $cats = array(
            'id'        => 'knowledge_base',
            'menu_type' => (Singular_Core::_('theme')->getDefault() == 'mobile') ? 'top-menu' : 'top-bar',
            'label'     => self::getView()->t('Knowledge Base'),
            'order'     => 51,
            'badge'     => null,
            'route'     => 'knowledgebase_default'
        );

        /** @var $categories Zend_Db_Table_Rowset */
        $categories = Singular_Loader::get('KnowledgeBase_Model_DbTable_Categories')->findAll();

        if ($categories->valid()) {
            /** @var $category KnowledgeBase_Model_Category */
            foreach ($categories as $category) {
                if ((bool) !$category->countResponses()) continue;

                $cats['pages'][$category->slug] = array(
                    'id'     => 'knowledge-base-' . $category->slug,
                    'class'  => 'knowledge-base',
                    'label'  => self::getView()->escape($category->name),
                    'route'  => 'knowledgebase_category',
                    'params' => array(
                        'category_slug' => $category->slug
                    ),
                    'pages'  => array()
                );

                /** @var $subcategories Zend_Db_Table_Rowset */
                $subcategories = $category->getAllSubcategories();
                if ($subcategories && (bool) count($subcategories)) {
                    /** @var $category KnowledgeBase_Model_Category */
                    foreach ($subcategories as $subcategory) {
                        if ((bool) !$subcategory->countResponses()) continue;
                        $cats['pages'][$category->slug]['pages'][$subcategory->slug] = array(
                            'id'            => 'knowledge-base-' . $subcategory->slug,
                            'class'         => 'knowledge-base',
                            'label'         => self::getView()->escape($subcategory->name),
                            'route'         => 'knowledgebase_category',
                            'hide_in_bread' => true,
                            'params'        => array(
                                'category_slug' => $subcategory->slug
                            )
                        );
                    }
                }
            }
        }

        // Breadcrumbs on Knowledge Base response show
        $route = Zend_Controller_Front::getInstance()->getRouter()->getCurrentRouteName();
        if ('knowledgebase_response' == $route) {
            /** @var $response System_Model_Response */
            $response = Singular_Runtime::extract('activeResponse');

            $cats['pages'][$response->slug] = array(
                'id'     => 'knowledge-base-' . $response->slug,
                'class'  => 'knowledge-base',
                'label'  => self::getView()->truncate(self::getView()->escape($response->title), 30),
                'route'  => 'knowledgebase_response',
                'hide' => true,
                'params' => array(
                    'response_slug' => $response->slug
                ),
                'pages'  => array()
            );
        }

        $page = $navigation->findBy('id', 'main-page');
        $page->addPage($cats);

        // Admin navigation
        if ($loggedUser->isAdmin() || $loggedUser->isAllowed('can_manage_knowledge-base')) {
            $page = $navigation->findBy('id', 'admin');
            if ($page instanceof Zend_Navigation_Page) {
                $page->addPage(array(
                    'id'         => 'knowledge-base-manage',
                    'menu_type'  => 'admin-sidebar',
                    'order'      => 35,
                    'image'      => self::getView()->imageSrc('icon_small.png', 'knowledge-base', false),
                    'label'      => self::getView()->t('Knowledge Base'),
                    'route'      => 'admin_knowledgebase',
                    'pages'      => array()
                ));
            }
        }
    }

    /**
     * Subscribes to "formInitAfter" event.
     *
     * @param  System_Form_Response $form
     *
     * @return void
     */
    public static function onFormInitAfter($form)
    {
        // Exclude from widget form
        $currentRoute = Zend_Controller_Front::getInstance()->getRouter()->getCurrentRouteName();
        if ('responses_widget' == $currentRoute) return;

        // Permission check
        $loggedUser = Singular_Runtime::extract("loggedUser");
        if (!$loggedUser->isAdmin() && !$loggedUser->isAllowed('can_manage_knowledge-base')) return;

        // Add Knowledge Base Category select to Response add/edit form
        if ($form instanceof System_Form_Response) {

            // Don't apply on "mobile" theme
            if (Singular_Core::_('Theme')->getDefault() == 'mobile') {
                return;
            }

            $response = self::getView()->activeResponse;
            if (!$response instanceof System_Model_Response) return;
            $categoriesModel = new KnowledgeBase_Model_DbTable_Categories();
            $categories = $categoriesModel->findAll(
                $categoriesModel->select()->where('parent_id = ?', '0')
            );

            if ((bool) !count($categories)) return;

            $knowledgeBase = new Zend_Form_Element_Select('knowledgeBase',
                array(
                     'required'    => false,
                     'label'       => 'Knowledge Base',
                     'order'       => 0,
                     'categories'  => $categories,
                     'description' => 'Assign response to Knowledge Base and select category',
                     'decorators'  => array(
                         array('ViewScript',
                               array('viewScript' => 'forms/elements/categories-tree.phtml')
                         ),
                     ),
                )
            );

            $knowledgeBase->setRegisterInArrayValidator(false);

            $knowledgeBaseEnable = new Zend_Form_Element_Hidden('knowledgeBaseEnable', array(
                'decorators' => array(
                    'ViewHelper'
                ),
                'value' => '0'
            ));

            if (!$response->isNew()) {
                $category = $categoriesModel->findByResponseId($response->id);
                if ($category->valid()) {
                    $category = $category->current();
                    $knowledgeBase->setValue($category->id);
                    $knowledgeBaseEnable->setValue(1);
                }
            }

            $group = $form->getDisplayGroup('default');
            if (empty($group)) {
                $form->addDisplayGroup(array($knowledgeBase, $knowledgeBaseEnable), 'default');
            } else {
                $group->addElement($knowledgeBase, 'knowledgeBase');
                $group->addElement($knowledgeBaseEnable, 'knowledgeBaseEnable');
            }

            self::getView()->additionalModulePath('knowledge-base');

        // Add Knowledge Base option to Home Page Content
        } elseif ($form instanceof System_Form_ResponsesSettings) {

            if (!$form->homepage_content_knowledge_base) {
                $kbHomepageSetting = Singular_Core::_('Settings')->getSetting('homepage_content_knowledge_base', true);
                $element = new Zend_Form_Element_Checkbox('homepage_content_knowledge_base',
                    array(
                         'value'       => $kbHomepageSetting->value,
                         'label'       => $kbHomepageSetting->title,
                         'allowEmpty'  => false,
                         'filters'     => array('StringTrim'),
                         'validators'  => array(
                             array('validator' => 'NotEmpty', 'options' => array(
                                 'messages' => array(
                                     Zend_Validate_NotEmpty::IS_EMPTY => '"' . $kbHomepageSetting->title . '" is required!'
                                 )
                             ))
                         )
                    )
                );
                $element->setDecorators(
                    array(
                         array('ViewScript', array('viewScript' => 'forms/elements/checkbox.phtml')),
                         'ViewHelper'
                    )
                );

                if (!$form->getDisplayGroup('homepage_content')) {
                    $form->addDisplayGroup(array($element), 'homepage_content');
                } else {
                    $form->getDisplayGroup('homepage_content')->addElement($element);
                }
            }
        }
    }

    /**
     * Subscribes to "afterSave" event.
     *
     * @param  System_Model_Response $object
     *
     * @return void
     */
    public static function onAfterSave($object)
    {
        // Permission check
        $loggedUser = Singular_Runtime::extract("loggedUser");
        if (!$loggedUser->isAdmin() && !$loggedUser->isAllowed('can_manage_knowledge-base')) return;

        // Process Knowledge Base assignment on response add/edit or Api request
        if ($object instanceof System_Model_Response) {

            // Don't apply on "mobile" theme
            if (Singular_Core::_('Theme')->getDefault() == 'mobile') {
                return;
            }

            $front        = Zend_Controller_Front::getInstance();
            $currentRoute = $front->getRouter()->getCurrentRouteName();
            $request      = $front->getRequest();
            $isApiCall    = $request->getControllerName() == 'api'
                && $request->getModuleName() == 'system'
                && ($request->getActionName() == 'create-response' || $request->getActionName() == 'change-response');

            if ($currentRoute == 'response_add' || $currentRoute == 'response_edit' || $isApiCall) {
                if ($request->isPost()) {

                    $categoryId = $request->getParam('knowledgeBase', null);
                    $kbResponsesModel = Singular_Loader::get('KnowledgeBase_Model_DbTable_Responses');

                    $response = null;
                    //if (!$object->isNew()) {
                        $response = $kbResponsesModel->fetchRow(
                            $kbResponsesModel->select()->where('response_id = ?', $object->id)
                        );
                    //}

                    if (is_null($categoryId)) {
                        if ($response instanceof KnowledgeBase_Model_Response) {
                            $response->delete();
                        }
                        return;
                    }

                    /** @var $kbCategoriesModel KnowledgeBase_Model_DbTable_Categories */
                    $kbCategoriesModel = Singular_Loader::get(
                        'KnowledgeBase_Model_DbTable_Categories'
                    );

                    /** @var $category KnowledgeBase_Model_Category */
                    $category = $kbCategoriesModel->fetchRow(
                        $kbCategoriesModel->select()->where('id = ?', $categoryId)
                    );

                    // Deletes existed Response record if Knowledge Base assignment is switched off or in trash
                    if ((bool) !$request->getParam('knowledgeBaseEnable') || 0 != $object->deleted_by_id) {
                        if ($response instanceof KnowledgeBase_Model_Response) {
                            $response->delete();
                            // Unlock comments
                            $object->lock_comments = 0;
                        }
                        return;
                    // When Response category is not changed - Stop operation
                    } elseif ($response && $categoryId == $response->category_id) {
                        if (!$category->isSubcategory()) {
                            $response->category_id    = $categoryId;
                            $response->subcategory_id = $categoryId;
                            $response->save();
                        }
                        return;
                    // When category has changed - Delete old record before adding new record
                    } elseif ($response) {
                        $response->delete();
                    }

                    unset($response);

                    // Create new Response category record
                    $response = new KnowledgeBase_Model_Response();
                    $response->setFromArray(
                        array('response_id'    => $object->id,
                              'category_id'    => ($category->isSubcategory()) ? $category->parent_id : $category->id,
                              'subcategory_id' => $category->id)
                    );
                    $response->save();

                    // Lock comments
                    $object->lock_comments = 1;
                    // Remove private if set
                    if (isset($object->is_private) && (bool) $object->is_private) {
                        $object->is_private = 0;
                    }
                }
            }
        }
    }

    /**
     * Subscribes to "viewVarSet" event
     * Directly assigns a variable to the view script.
     *
     * @param stdClass $data
     */
    public static function onViewVarSet ($data)
    {
        $route         = Zend_Controller_Front::getInstance()->getRouter()->getCurrentRouteName();
        $allowedRoutes = array('responses', 'responses_type', 'responses_show', 'responses_show_old',
                               'knowledgebase_default', 'knowledgebase_category', 'knowledgebase_search');

        if ('types' != $data->name || !$data->value instanceof System_Model_Rowset || !in_array($route, $allowedRoutes)
        ) return;
        // Prevent for adding twice
        if (count($data->value->findItemsByField('slug', 'knowledge_base'))) return;

        // Add Knowledge Base to Responses types list
        $data->value->add(
            array(
                 'slug'              => 'knowledge_base',
                 'title'             => self::getView()->t('Knowledge Base'),
                 'objects'           => 'knowledge-base',
                 'icon'              => 'icon_medium.png',
                 'module_id'         => Singular_Loader::get('System_Model_DbTable_Modules')
                                            ->cache_findByName('knowledge-base')->id,
                 'has_comments'      => 1,
                 'has_negative_vote' => 1,
                 'has_votes'         => 0,
                 'is_active'         => 1,
                 'is_default'        => 0,
                 'order'             => 0,
                 'route'             => array(
                     'name'   => 'knowledgebase_default',
                     'params' => array()
                 )
            ), 1
        );
    }

    /**
     * Subscribes to "select" event
     *
     * @param  Zend_Db_Table_Select $select
     *
     * @return void
     */
    public static function onSelect (Zend_Db_Table_Select $select)
    {
        $table          = $select->getTable();
        $front          = Zend_Controller_Front::getInstance();
        $routeName      = $front->getRouter()->getCurrentRouteName();
        /*
         * Removed from affected routes:  'search', 'quick_search'
         * to include knowledge-base responses into search procedure.
         */
        $affectedRoutes = array('default', 'responses', 'responses_type',
                                'responses_show', 'responses_show_old',
                                'responses_widget',
                                'profile_me', 'profile', 'categories',
                                'categories_category');
        $excludedRoutes = array('response_add', 'response_add_reload_captcha',
                                'response_add_tag',
                                'response_change_status',
                                'response_lock_comments', 'response_add_answer',
                                'response_edit_answer', 'response_edit',
                                'response_trash', 'response_delete',
                                'response_recovery', 'response_set-order',
                                'response_vote_add', 'response_rss');

        $condition = ((in_array($routeName, $affectedRoutes)
            && !Singular_Runtime::inStorage('knowledgeBase'))
            && !in_array($routeName, $excludedRoutes)
            && ($table instanceof System_Model_DbTable_Responses
                || $table instanceof
                    Categories_Model_DbTable_ResponsesCategories));

        if ($condition) {
            $tableName          = $select->getTable()->getTableName();
            $responsesTableName = Singular_Loader::get(
                "KnowledgeBase_Model_DbTable_Responses"
            )->getTableName();
            $joinFiled          = 'id';

            if ($table instanceof Categories_Model_DbTable_ResponsesCategories) {
                $joinFiled = 'response_id';
            }
            $select->joinLeft(
                array('kbr' => $responsesTableName),
                "$tableName.$joinFiled=kbr.response_id ",
                array()
            );

            $select->where("kbr.response_id IS NULL");
        }
    }

    /**
     * Subscribes to "componentRenderAfter" event
     *
     * @param  stdClass $data
     */
    public static function onComponentRenderAfter (stdClass $data)
    {
        if (!$data->component instanceof KnowledgeBase_Component_KnowledgeBase) return;
        if (Singular_Runtime::inStorage('knowledgeBase')) Singular_Runtime::remove('knowledgeBase');
    }

    /**
     * Called after Zend_Controller_Router exits.
     *
     * Called after Zend_Controller_Front exits from the router.
     *
     * @param  Zend_Controller_Request_Abstract $request
     *
     * @return void
     */
    public static function onRouteShutdown(Zend_Controller_Request_Abstract $request)
    {
        $router = Zend_Controller_Front::getInstance()->getRouter();
        if ('responses_show' == $router->getCurrentRouteName()) {

            /** @var $route Zend_Controller_Router_Route_Regex */
            $route = $router->getCurrentRoute();

            // Get response slug
            $refObj = new ReflectionObject($route);
            $values = $refObj->getProperty('_values');
            $values->setAccessible(true); // Gather access to this properties value, although it is private
            $responseSlug = current($values->getValue($route));

            /** @var $kbResponsesModel KnowledgeBase_Model_DbTable_Responses */
            $kbResponsesModel = Singular_Loader::get('KnowledgeBase_Model_DbTable_Responses');

            $responses = $kbResponsesModel->getResponsesByCond(
                array('r.slug = ?' => $responseSlug)
            );

            // Redirect to Knowledge Base article
            if ((bool) count($responses)) {
                // Knowledge base article url
                Zend_Controller_Action_HelperBroker::getStaticHelper('redirector')
                    ->setCode(301)
                    ->setGotoRoute(array('response_slug' => $responseSlug), 'knowledgebase_response');
            }
        }
    }

    /**
     * Subscribes to "onMainPage" event.
     *
     * @param  stdClass $data
     *
     * @return void
     */
    public static function onMainPage($data)
    {
        if (1 != Singular_Core::_("Settings")->getSetting('homepage_content_knowledge_base')) {
            return;
        }

        // Don't apply on "mobile" theme
        if (Singular_Core::_('Theme')->getDefault() == 'mobile') {
            return;
        }

        $cacheId = Singular_Runtime::extract('loggedUser')->role . "homepage_knowledge_base";
        $cacheId = preg_replace("/[^a-zA-Z0-9_]/", "", $cacheId);
        $tags    = null;
        $cache   = Singular_Runtime::extract('cachemanager')->getCache('default');

        if (($output = $cache->load($cacheId)) == false) {

            $notEmpty = (bool) Singular_Loader::get
                ('KnowledgeBase_Model_DbTable_Responses')->count();
            if (!$notEmpty) {
                return;
            }

            self::getView()->cacheTags       = new StdClass();
            self::getView()->cacheTags->tags = array(
                'knowledge_base',
                'responses',
                'response'
            );

            $output = Singular_Component::run(
                'KnowledgeBase', 'knowledge-base',
                array('variant'       => 'default', 'widget' => true,
                      'displaySearch' => false)
            );

            $tags   = self::getView()->cacheTags->tags;
            $tags[] = 'responses';
            $cache->save($output, $cacheId, $tags);
        }

        $data->sections[1] = $output;
    }

    /**
     * Subscribes to "moduleDeactivateBefore" event.
     *
     * @param System_Model_Module $module
     */
    public static function onModuleDeactivateBefore(System_Model_Module $module)
    {
    }

    /**
     * Subscribes to "moduleUninstallBefore" event.
     *
     * @param System_Model_Module $module
     */
    public static function onModuleUninstallBefore(System_Model_Module $module)
    {
    }

    /**
     * Subscribes to "postDispatchOnce" event.
     *
     * @param  Zend_Controller_Request_Abstract $request
     * @return void
     */
    public static function onPostDispatchOnce ($request)
    {
        if (System_Service_Environment::getInstance()->isFrontend()) {
            self::getView()->headLink()->prependStylesheet(
                self::getView()->stylesheetSrc('frontend.css', 'knowledge-base')
            );
            self::getView()->headScript()->appendFile(
                self::getView()->javascriptSrc('main.js', 'knowledge-base')
            );
        }

        $routeName = Zend_Controller_Front::getInstance()->getRouter()->getCurrentRouteName();
        if ($routeName == 'admin_widget') {
            self::getView()->headScript()->appendFile(
                self::getView()->javascriptSrc('admin.js', 'knowledge-base')
            );
            self::getView()->headLink()->prependStylesheet(
                self::getView()->stylesheetSrc('admin.css', 'knowledge-base')
            );
        }
    }

    /**
     * Subscribes to "widgetStack" event.
     *
     * Builds widget stack
     *
     * @param  SplPriorityQueue $widgetStack
     */
    public static function onWidgetStack(SplPriorityQueue $widgetStack)
    {
        if (System_Service_Environment::getInstance()->isAdmin()) {
            $front = Zend_Controller_Front::getInstance();
            $routeName = $front->getRouter()->getCurrentRouteName();
            if ($routeName == 'admin_widget') {
                self::getView()->additionalModulePath('knowledge-base');
                self::getView()->knowledgeBaseWidgetForm = new KnowledgeBase_Form_Widget(
                    self::getView()->settings->{'knowledge-base'}->widget
                );

                $widgetStack->insert(
                    array(
                         'alias'   => 'knowledge-base',
                         'title'   => self::getView()->t('Knowledge Base'),
                         'content' => self::getView()->render('knowledge-base_admin_widget.phtml')
                    ), 80
                );
            }
        }
    }
}
