<?php
namespace Useresponse\Sdk;

use Useresponse\Sdk\Adapters\IAdapter;
use Useresponse\Sdk\Services\ServiceAbstract;

/**
 * Class Sdk
 *
 * @package Useresponse\Sdk
 */
class Sdk
{

    const FORMAT_JSON = 'json';
    const FORMAT_DEFAULT = self::FORMAT_JSON;

    /**
     * Sdk instances.
     *
     * @var Sdk[]
     */
    protected static $instances = [];

    /**
     * Allowed response formats.
     *
     * @var array
     */
    protected static $allowedFormats = [self::FORMAT_JSON];

    /**
     * API adapter.
     *
     * @var IAdapter
     */
    protected $adapter;

    /**
     * Response format.
     *
     * @var string
     */
    protected $format = self::FORMAT_DEFAULT;

    /**
     * Services.
     *
     * @var ServiceAbstract[]
     */
    protected $services = [];

    /**
     * SDK constructor.
     *
     * @param array $config
     * @throws \Exception
     */
    protected function __construct(array $config)
    {
        if (empty($config['apiSource'])) {
            throw new \Exception("Config option 'apiSource' is required!");
        }

        if (empty($config['adapter'])) {
            throw new \Exception("Config option 'adapter' is required!");
        }

        if (!empty($config['format'])) {
            if (!in_array($config['format'], self::$allowedFormats)) {
                throw new \Exception("Specified response format is not allowed!");
            }

            $this->format = $config['format'];
        } else {
            $config['format'] = self::FORMAT_DEFAULT;
        }

        $adapterName = $config['adapter'];
        $adapterClass = 'Useresponse\Sdk\Adapters\\' . $adapterName;

        if (!is_readable(__DIR__ . "/Adapters/{$adapterName}.php")) {
            throw new \Exception("Adapter '{$adapterName}' doesn't exists!");
        }

        $this->adapter = new $adapterClass($config);
    }

    /**
     * Get API service.
     *
     * @param string $name
     * @return ServiceAbstract
     * @throws \Exception
     */
    public function getService($name)
    {
        if (empty($this->services[$name])) {
            if (!is_readable(__DIR__ . "/Services/{$name}.php")) {
                throw new \Exception("Service '{$name}' doesn't exists!");
            }

            $serviceClass = 'Useresponse\\Sdk\\Services\\' . $name;
            $this->services[$name] = new $serviceClass($this->adapter, $this->format);
        }

        return $this->services[$name];
    }

    /**
     * Returns sdk adapter.
     *
     * @return IAdapter
     */
    public function adapter()
    {
        return $this->adapter;
    }

    /**
     * Returns encoding format.
     *
     * @return string
     */
    public function format()
    {
        return $this->format;
    }

    /**
     * Initialize new SDK instance.
     *
     * @param string $alias
     * @param array $config
     * @return static
     * @throws \Exception
     */
    public static function init($alias, array $config)
    {
        if (empty($alias)) {
            throw new \Exception("Alias is required!");
        }

        return self::$instances[$alias] = new self($config);
    }

    /**
     * Returns true if sdk registered.
     *
     * @param string $alias
     * @return bool
     */
    public static function exists($alias)
    {
        return array_key_exists($alias, self::$instances);
    }

    /**
     * Get SDK instance by alias.
     *
     * @param string $alias
     * @return Sdk
     * @throws \Exception
     */
    public static function get($alias)
    {
        if (!self::exists($alias)) {
            throw new \Exception("SDK instance with alias '{$alias}' doesn't exist!");
        }

        return self::$instances[$alias];
    }
}