<?php

namespace Useresponse\Sdk;

/**
 * Class Response
 *
 * @package Useresponse\Sdk
 */
class Response
{

    /**
     * Response error.
     *
     * @var \stdClass
     */
    protected $error;

    /**
     * API result.
     *
     * @var \stdClass
     */
    protected $result;

    /**
     * Decoder name.
     *
     * @var string
     */
    protected $decoderName;

    /**
     * Allowed HTTP codes.
     *
     * @var array
     */
    protected $allowedHttpCodes = [200, 201];

    /**
     * Response constructor.
     *
     * @param \stdClass $result
     * @param string $format
     */
    public function __construct(\stdClass $result, $format)
    {
        $this->result = $result;
        $this->decoderName = ucfirst($format);
    }

    /**
     * Fetch result.
     *
     * @return mixed
     * @throws \Exception
     */
    public function fetch()
    {
        $response = null;

        if ($this->isError()) {
            $response = $this->error;
        } else {
            if (!is_readable(__DIR__ . "/Decoders/{$this->decoderName}.php")) {
                throw new \Exception("Decoder '{$this->decoderName}' doesn't exists!");
            }

            $decoderClass = '\Useresponse\Sdk\Decoders\\' . $this->decoderName;
            $decoder = new $decoderClass($this->result->content);

            if (!($decoder instanceof \Useresponse\Sdk\Decoders\DecoderAbstract)) {
                throw new \Exception("Decoder should extend 'DecoderAbstract'!");
            }

            $response = $decoder->decode();
            $decodeError = $decoder->getError();

            if ($decodeError) {
                $response = (object)[
                    'code' => 0,
                    'error' => $decodeError
                ];
            }
        }

        return $response;
    }

    /**
     * Detect errors.
     *
     * @return bool
     */
    protected function isError()
    {
        $this->error = null;

        if ($this->result->errno) {
            $this->error = (object)[
                'code' => 0,
                'error' => $this->result->error
            ];
        } elseif (!in_array($this->result->httpCode, $this->allowedHttpCodes)) {
            $this->error = (object)[
                'code' => $this->result->httpCode,
                'error' => $this->result->content
            ];
        }

        return is_object($this->error);
    }

}