<?php
namespace Useresponse\Sdk\Services;

use Useresponse\Sdk\Adapters\IAdapter;

/**
 * Class Objects
 *
 * @package Useresponse\Sdk\Services
 */
class Objects extends ServiceAbstract
{

    /**
     * Fetch object by id.
     *
     * @param int $id
     * @return mixed
     * @throws \Exception
     */
    public function getObject($id)
    {
        if (!isset($id)) {
            throw new \Exception("Object id is required!");
        }

        $result = $this->adapter->request('objects/' . (int)$id, IAdapter::TYPE_GET);

        return $this->makeResponse($result);
    }

    /**
     * Fetch object by slug.
     *
     * @param string $slug
     * @return mixed
     * @throws \Exception
     */
    public function getObjectBySlug($slug)
    {
        if (empty($slug)) {
            throw new \Exception('Object slug is required!');
        }

        $result = $this->adapter->request('objects/' . $slug, IAdapter::TYPE_GET);

        return $this->makeResponse($result);
    }

    /**
     * Fetch objects.
     *
     * @param array $data
     * @return mixed
     */
    public function getObjects(array $data = [])
    {
        $result = $this->adapter->request('search', IAdapter::TYPE_GET, $data);

        return $this->makeResponse($result);
    }

    /**
     * Fetch tickets.
     *
     * @param array $data
     * @return mixed
     */
    public function getTickets(array $data = [])
    {
        $result = $this->adapter->request('tickets', IAdapter::TYPE_GET, $data);

        return $this->makeResponse($result);
    }

    /**
     * Fetch my objects.
     *
     * @return mixed
     */
    public function getMyObjects()
    {
        $result = $this->adapter->request('profile/objects', IAdapter::TYPE_GET);

        return $this->makeResponse($result);
    }

    /**
     * Add new object.
     *
     * @param array $data
     * @return mixed
     * @throws \Exception
     */
    public function addObject(array $data)
    {
        $result = $this->adapter->request('objects', IAdapter::TYPE_POST, $data);

        return $this->makeResponse($result);
    }

    /**
     * Edit object by id.
     *
     * @param int $id
     * @param array $data
     * @return mixed
     * @throws \Exception
     */
    public function editObject($id, array $data)
    {
        if (!isset($id)) {
            throw new \Exception("Object id is required!");
        }

        $result = $this->adapter->request('objects/' . (int)$id, IAdapter::TYPE_PUT, $data);

        return $this->makeResponse($result);
    }

    /**
     * Delete object by id.
     *
     * @param int $id
     * @return mixed
     * @throws \Exception
     */
    public function deleteObject($id)
    {
        if (!isset($id)) {
            throw new \Exception("Object id is required!");
        }

        $result = $this->adapter->request('objects/' . (int)$id, IAdapter::TYPE_DELETE);

        return $this->makeResponse($result);
    }

    /**
     * Move object to trash by id.
     *
     * @param int $id
     * @return mixed
     * @throws \Exception
     */
    public function moveToTrash($id)
    {
        if (!isset($id)) {
            throw new \Exception("Object id is required!");
        }

        $result = $this->adapter->request('objects/' . (int)$id . '/trash', IAdapter::TYPE_GET);

        return $this->makeResponse($result);
    }

    /**
     * Move object to archive by id.
     *
     * @param int $id
     * @return mixed
     * @throws \Exception
     */
    public function moveToArchive($id)
    {
        if (!isset($id)) {
            throw new \Exception("Object id is required!");
        }

        $result = $this->adapter->request('objects/' . (int)$id . '/archive', IAdapter::TYPE_GET);

        return $this->makeResponse($result);
    }

    /**
     * Recover object by id.
     *
     * @param int $id
     * @return mixed
     * @throws \Exception
     */
    public function recover($id)
    {
        if (!isset($id)) {
            throw new \Exception("Object id is required!");
        }

        $result = $this->adapter->request('objects/' . (int)$id . '/recovery', IAdapter::TYPE_GET);

        return $this->makeResponse($result);
    }

    /**
     * Toggle object commenting by id.
     *
     * @param int $id
     * @return mixed
     * @throws \Exception
     */
    public function toggleCommenting($id)
    {
        if (!isset($id)) {
            throw new \Exception("Object id is required!");
        }

        $result = $this->adapter->request('objects/' . (int)$id . '/comments/toggle', IAdapter::TYPE_GET);

        return $this->makeResponse($result);
    }

    /**
     * Vote for object by id.
     *
     * @param int $id
     * @param string $value
     * @return mixed
     * @throws \Exception
     */
    public function voting($id, $value)
    {
        if (!isset($id)) {
            throw new \Exception("Object id is required!");
        }

        if (!in_array($value, ['up', 'down', 'revert'])) {
            throw new \Exception("Value can be 'up', 'down' or 'revert'!");
        }

        $result = $this->adapter->request('objects/' . (int)$id . '/votes/toggle', IAdapter::TYPE_GET, ['value' => $value]);

        return $this->makeResponse($result);
    }

    /**
     * Subscribe on object by id.
     *
     * @param int $id
     * @param int $userId
     * @return mixed
     * @throws \Exception
     */
    public function subscribe($id, $userId = null)
    {
        if (!isset($id)) {
            throw new \Exception("Object id is required!");
        }

        $data = $userId === null ? [] : ['user_id' => $userId];
        $result = $this->adapter->request('objects/' . (int)$id . '/subscribe', IAdapter::TYPE_GET, $data);

        return $this->makeResponse($result);
    }

    /**
     * Unsubscribe from object by id.
     *
     * @param int $id
     * @param int $userId
     * @return mixed
     * @throws \Exception
     */
    public function unsubscribe($id, $userId = null)
    {
        if (!isset($id)) {
            throw new \Exception("Object id is required!");
        }

        $data = $userId === null ? [] : ['user_id' => $userId];
        $result = $this->adapter->request('objects/' . (int)$id . '/unsubscribe', IAdapter::TYPE_GET, $data);

        return $this->makeResponse($result);
    }

    /**
     * List custom fields.
     *
     * @return mixed
     */
    public function listCustomFields()
    {
        $result = $this->adapter->request('objects/custom-fields/list');

        return $this->makeResponse($result);
    }
}