<?php
/**************************************************************
 * Copyright USWebStyle Inc., 2011-2012. All Rights Reserved.
 **************************************************************
 * NOTICE:  This source code is the property of USWebStyle, Inc.
 * In no case shall you rent, lease, lend, redistribute
 * any of below source code to a 3rd party individual or entity.
 *
 * @category   Loginplus
 * @package    LoginPlus_Handler
 * @copyright  USWebStyle Inc., 2011-2012 (http://www.uswebstyle.com)
 * @license    http:/www.useresponse.com/licensing (Commercial)
 */

/**
 * LoginPlus handlers
 *
 * @uses       Singular_Event_Listener_Abstract
 * @category   Loginplus
 * @package    LoginPlus_Handler
 */
class Loginplus_Handler_Events extends Singular_Event_Listener_Abstract
{
    /**
     * Binds event listeners to the dispatcher.
     * Must be implemented as interface convention.
     *
     * @param  Singular_Event_Dispatcher $dispatcher
     * @return void
     */
    public static function bindListeners()
    {
        self::attachAggregate('preDispatch', array(__CLASS__, 'onPreDispatch'));
        self::attachAggregate('authSidebar', array(__CLASS__, 'onAuthSidebar'));
        self::attachAggregate('authForms', array(__CLASS__, 'onAuthForms'));
        self::attachAggregate('postDispatchOnce', array(__CLASS__, 'onPostDispatchOnce'));
        self::attachAggregate('buildNavigation', array(__CLASS__, 'onNavigation'));
        self::attachAggregate('afterRegistration', array(__CLASS__, 'onAfterRegistration'), self::PRIORITY_HIGH);
    }

    public static function onAuthForms($data)
    {
        $method = Singular_Core::_('settings')->getSetting('sso_active_method');
        if ($method == 'ldap') {
            $data->forms[] = array(
                'id' => 'ldap',
                'form' => new Loginplus_Form_LdapLogin()
            );
        }
    }

    /**
     * Subscribes to "afterRegistration" event.
     *
     * @param  System_Model_User $user
     * @return void
     */
    public static function onAfterRegistration ($user)
    {
        $isRegNotifyDisabled = Singular_Core::_('settings')->getSetting('sso_reg_notification_disabled');
        if (!is_null($isRegNotifyDisabled) && $isRegNotifyDisabled == '1') {
            Singular_Event::stopPropagation('afterRegistration');
        }
    }    

    /**
     * Subscribes to "preDispatch" event.
     *
     * @param  Zend_Controller_Request_Abstract $request
     * @return void
     */
    public static function onPreDispatch(Zend_Controller_Request_Abstract $request)
    {
        $settings   = Singular_Core::_("settings");
        $loggedUser = Singular_Runtime::extract("loggedUser");
        if ($settings->getSetting('sso_native_auth_disabled') == 1
            && $settings->getSetting('sso_active_method') != 'none'
            && $loggedUser->isGuest()
        ) {
            Singular_AppClient::setParam('require_sso_login', 1);
        }

        $logoutUrl = Singular_Core::_('settings')->getSetting('sso_onelogin_logouturl');
        $loginUrl = Singular_Core::_('settings')->getSetting('sso_onelogin_loginurl');

        switch($request->getParam('action')) {
            case 'logout':
                self::_ssoDeleteCookie();
                if (!empty($logoutUrl)) {
                    $ssoActiveMethod = Singular_Core::_('settings')->getSetting('sso_active_method');
                    if ($ssoActiveMethod == 'integration') {
                        Zend_Auth::getInstance()->clearIdentity();
                        header("Location: {$logoutUrl}");
                        exit;
                    }
                }
                break;

            default:
                self::_integrationReadCookie();
                self::_ldapReadCookie();
                break;

        }
    }

    /**
     * Subscribes to "postDispatchOnce" event.
     *
     * @param  Zend_Controller_Request_Abstract $request
     * @return void
     */
    public static function onPostDispatchOnce(Zend_Controller_Request_Abstract $request)
    {
        self::getView()->headScript()->appendFile(self::getView()->javascriptSrc('main.js', 'loginplus'));
        self::getView()->headLink()->prependStylesheet(self::getView()->stylesheetSrc('common.css', 'loginplus'));

        if (!Zend_Auth::getInstance()->hasIdentity() && (
            (Singular_Core::_("settings")->getSetting('site_twitter_app_id')
                && Singular_Core::_("settings")->getSetting('site_twitter_app_secret'))
                || (Singular_Core::_("settings")->getSetting('site_facebook_app_id')
                && Singular_Core::_("settings")->getSetting('site_facebook_app_secret'))
                || (Singular_Core::_("settings")->getSetting('site_google_app_id')
                && Singular_Core::_("settings")->getSetting('site_google_app_secret'))
        )) {
            self::getView()->headLink()->prependStylesheet(self::getView()->stylesheetSrc('frontend.css', 'loginplus'));
            switch($request->getParam('action')) {
                case 'logout':
                    if (!empty($logoutUrl)) {
                        header("Location: {$logoutUrl}");
                        exit;
                    }
                    break;
            }

        }
    }
    
    /**
     * Subscribes to "navigation" event.
     *
     * Builds navigation
     *
     * @param  Singular_Navigation $navigation
     * @return void
     */
    public static function onNavigation($navigation)
    {
        $isNativeAuthDisabled = Singular_Core::_('settings')->getSetting('sso_native_auth_disabled');
        $ssoActiveMethod      = Singular_Core::_('settings')->getSetting('sso_active_method');

        if (!is_null($isNativeAuthDisabled) && $isNativeAuthDisabled == '1' && 'none' != $ssoActiveMethod) {

            $loginUrl = Singular_Core::_('settings')->getSetting('sso_onelogin_loginurl');
            $loginMenuItem = $navigation->findBy('id', 'auth-login');
            if (!is_null($loginMenuItem)) {
                $loginMenuItem->route = 'sso-loginurl';
                $loginMenuItem->class = 'login';
            }

            $registerMenuItem = $navigation->findBy('id', 'auth-register');
            if (!is_null($registerMenuItem)) $registerMenuItem->getParent()->removePage($registerMenuItem);
            unset($registerMenuItem);

        }

        $loggedUser = Singular_Runtime::extract("loggedUser");
        if ($loggedUser->isAdmin()) {
            $page = $navigation->findBy('id', 'admin');

            if ($page instanceof Zend_Navigation_Page) {
                $page->addPage(array(
                    'id' => 'login-plus',
                    'menu_type' => 'admin-sidebar',
                    'order' => 80,
                    'image' => self::getView()->imageSrc('icon_small.png', 'loginplus', false),
                    'label' => self::getView()->t('Login Plus'),
                    'route' => 'login_plus_admin',
                    'pages' => array(
                        array(
                            'id'         => 'login-plus-sso',
                            'menu_type'  => 'admin-sidebar',
                            'class'      => 'gray',
                            'label'      => self::getView()->t('Single Sign-On'),
                            'order'      => 81,
                            'route'      => 'login_plus_sso_admin'
                        )
                    )
                ));
            }
        }

        if (!empty($ssoActiveMethod) && $ssoActiveMethod != 'none') {
            $ssoActiveMethod = "_{$ssoActiveMethod}PopupNavigation";
            self::$ssoActiveMethod($navigation);
        }
    }

    /**
     * Subscribes to "authSidebar" event.
     *
     * @param  stdClass $data
     * @return void
     */
    public static function onAuthSidebar($data)
    {
        self::getView()->additionalModulePath('loginplus');
        $data->sidebar[] = self::getView()->render('auth-sidebar.phtml');
    }

    /**
     * Adds SSO: OneLogin button to authorization popup navigation
     *
     * @param Zend_Navigation $navigation
     */
    protected static function _integrationPopupNavigation($navigation)
    {
        $popupMenu = $navigation->findBy('menu_type', 'login-register');
        $loginUrl= Singular_Core::_('settings')->getSetting('sso_onelogin_loginurl');
        $popupSsoLogin = new Zend_Navigation_Page_Uri();
        $ssoLogin = $popupSsoLogin->setUri("{$loginUrl}?sso_redirect=1");
        $ssoLogin->label = self::getView()->t('Single Sign-On');
        $ssoLogin->menu_type = 'login-register';
        $ssoLogin->id = 'sso-external-login';
        $ssoLogin->liClass="integrationTab";
        $ssoLogin->icoClass="isso";

        if ($popupMenu instanceof Zend_Navigation_Page) {
            $popupMenu->addPage($ssoLogin);
        }
    }

    /**
     * Adds SSO: Ldap button to authorization popup navigation
     *
     * @param Zend_Navigation $navigation
     */
    protected static function _ldapPopupNavigation($navigation)
    {
        $popupMenu = $navigation->findBy('menu_type', 'login-register');

        if ($popupMenu instanceof Zend_Navigation_Page) {
            $popupMenu->addPage(array(
                'id'         => 'ldap-login',
                'menu_type'  => 'login-register',
                'label'      => self::getView()->t('Single Sign-On'),
                'liClass'    => 'ldapTab',
                'icoClass'   => 'isso',
                'aRel'       => 'form-ldap'
            ));
        }
    }


    /**
     * Deletes OneLogin and LDAP auhtorization cookies
     */
    protected static function _ssoDeleteCookie()
    {
        setcookie('sso_user',  '', null, '/');
        setcookie('ldap_user', '', null, '/');
    }


    /**
     * Read OneLogin cookie.
     * Creates user if it doesn't exist already, or logs in if user found (by email)
     *
     * @throws Singular_Acl_Exception
     */
    protected static function _integrationReadCookie()
    {
        if ('integration' != Singular_Core::_('settings')->getSetting('sso_active_method')) {
            return;
        }

        $secret = Loginplus_Model_Integration::getSecret(
            Singular_Core::_('settings')->getSetting('sso_onelogin_domain')
        );

        $auth = Zend_Auth::getInstance()->getIdentity();
        if (!empty($auth)) return;
        $cookie = Zend_Controller_Front::getInstance()->getRequest()->getCookie('sso_user', '{}');
        if (!preg_match('/^\{.*\}$/', $cookie)) {
            return;
        }
        $ssoData = Zend_Json::decode($cookie);
        if (isset($ssoData['email'])) {
            $ssoData['email'] = Loginplus_Model_Integration::decrypt($ssoData['email'], $secret);
        }
        
        $testHash = Loginplus_Model_Integration::generateHash($ssoData, $secret);
        if (!isset($ssoData['hash']) || $ssoData['hash'] != $testHash) {
            return;
        }

        if (empty($ssoData)) return;

        $usersDb    = Singular_Loader::get("System_Model_DbTable_Users");
        $userObject = $usersDb->findByEmail($ssoData['email']);
        
        if ($userObject instanceof System_Model_User) {
            System_Service_Auth::loginByEmail($ssoData['email']);
        } else {
            
            $user = new System_Model_User();
            $register = $user->register(
                array(
                     'full_name' => $ssoData['fullname'],
                     'email'     => $ssoData['email'],
                )
            );
            if ($register) {
                System_Service_Auth::loginByEmail($user->email);
                Zend_Controller_Action_HelperBroker::getStaticHelper('redirector')->setGotoRoute(array(), 'default');

                return;
            }
        }
    }

    /**
     * Read LDAP cookie.
     * Creates user if it doesn't exist already, or logs in if user found (by email).
     * Dummy email (username@localhost) used when LDAP served didn't provide email address record for logged user.
     *
     * @return void;
     */
    protected static function _ldapReadCookie()
    {
        $auth = Zend_Auth::getInstance()->getIdentity();
        if (!empty($auth)) return;

        $cookie = Zend_Controller_Front::getInstance()->getRequest()->getCookie('ldap_user', '{}');
        if (!preg_match('/^\{.*\}$/', $cookie)) {
            return;
        }
        $ldapData = Zend_Json_Decoder::decode($cookie);

        if (empty($ldapData)) return;
        $usersDb    = Singular_Loader::get("System_Model_DbTable_Users");
        $userObject = $usersDb->findByEmail($ldapData['email']);

        if ($userObject instanceof System_Model_User) {
            System_Service_Auth::loginByEmail($ldapData['email']);
            $remember = Zend_Controller_Front::getInstance()->getRequest()->getCookie('ldap_user_remember', '');
            if (!empty($remember)) {
                Zend_Session::rememberMe(14*24*60*60); // 14 days
            }
        } else {
            $user = new System_Model_User();
            $register = $user->register(array(
                'full_name' => $ldapData['fullname'],
                'email' => $ldapData['email'],
            ));
            if ($register) {
                System_Service_Auth::loginByEmail($user->email);
                $remember = Zend_Controller_Front::getInstance()->getRequest()->getCookie('ldap_user_remember', '');
                if (!empty($remember)) {
                    Zend_Session::rememberMe(14*24*60*60); // 14 days
                }
                Zend_Controller_Action_HelperBroker::getStaticHelper('redirector')->setGotoRoute(array(), 'default');
                return;
            }
        }
    }

    /**
     * Authorize user in system by using email
     *
     * @param  string $email
     * @return bool
     * @todo dry
     */
    protected static function _loginByLdapId($ldapId)
    {
        $adapter = new Zend_Auth_Adapter_DbTable(Singular_Db_Table_Abstract::getDefaultAdapter());
        $tablePrefix = Singular_Runtime::extract('config')->resources->db->params->prefix;

        $adapter->setTableName($tablePrefix . 'users')
            ->setIdentityColumn('ldap_id')
            ->setCredentialColumn('ldap_id')
            ->setIdentity($ldapId)
            ->setCredential($ldapId)
            ->getDbSelect()->where('is_active = ?', 1);

        $auth = Zend_Auth::getInstance();

        if ($auth->authenticate($adapter)->isValid()) {
            $auth->getStorage()->write($adapter->getResultRowObject());
            return true;
        }

        return false;
    }
}
