<?php 
if (!defined('INSTALLATION')) exit('Bad Request.');
date_default_timezone_set('GMT');

/**
 * This file is part of the UseResponse.
 * UseResponse - Brilliant Customer Support http://www.useresponse.com/
 * @copyright  Copyright (c) 2011 USWebStyle, LLC. (http://www.uswebstyle.com)
 */
class Installation
{
    /**
     * Path to config
     * 
     * @var string
     */
    protected $_config = null;
    
    /**
     * Path to install config
     * 
     * @var string
     */
    protected $_installConfig = null;
    
    /**
     * Application root directory
     * 
     * @var string
     */
    protected $_rootPath = null;
    
    /**
     * Results storage
     * 
     * @var array
     */
    protected $_results = array();
    
    /**
     * Errors counet
     * 
     * @var int
     */
    protected $_errors = 0;
    
    /**
     * Warnings counter
     * 
     * @var int
     */
    protected $_warnings = 0;
    
    /**
     * Execute installation process
     * 
     * @var bool
     */
    protected $_isExecute = false;
    
    /**
     * Is valid everything?
     * 
     * @var bool
     */
    protected $_isValid = false;
    
    /**
     * Is uploaded license?
     * 
     * @var bool
     */
    protected $_isUploadedLicense = true;
    
    /**
     * Everything error
     * 
     * @var string
     */
    protected $_everythingError = null;
    
    /**
     * Database connection
     * 
     * @var Zend_Db_Adapter_Pdo_Mysql
     */
    protected $_connection = null;
    
    /**
     * Writable folders list
     * 
     * @var array
     */
    protected $_writableFolders = array(
        'application/configs',
        'application/sessions',
        'application/localization',
        'public/files',
        'public/files/avatars',
        'public/files/logo',
        'public/static',
        'cache',
        'work'
    );
    
    /**
     * Modules list for installation
     * 
     * @var array 
     */
    protected $_modules = array(
        1 => 'system',
        2 => 'resources',
        3 => 'ideas',
        4 => 'problems',
        5 => 'questions',
        6 => 'thanks'
    );
    
    /**
     * Required php extensions
     * 
     * @var array
     */
    protected $_requiredPhpExtensions = array(
        'gd' => 'Required for dynamic images rendering (avatars, file uploads, etc.)',
        'spl' => 'SPL is standard PHP extention that implements efficient data access interfaces and classes',
        'pcre' => 'Perl Compatible Regular Expressions is built-in PHP extention to parse regular expressions',
        'ctype' => 'Checks whether a character or string falls into a certain character class according to the current locale',
        'libxml' => 'Base PHP extention that provides core functionality for parsing XML documents',
        'reflection' => 'Complete reflection API that adds the ability to reverse-engineer classes, interfaces, functions, methods and extensions',
        'pdo_mysql' => 'Lightweight, consistent interface for accessing databases in PHP',
        'dom' => 'The DOM extension allows you to operate on XML documents through the DOM API with PHP 5.x',
        'json' => 'JavaScript Object Notation (JSON) is a lightweight text-based open standard designed for human-readable data interchange',
        'iconv' => 'With this module, you can turn a string represented by a local character set into the one represented by another character set, which may be the Unicode character set',
        'xml' => 'This toolkit lets you parse, but not validate, XML documents',
    );
    
    /**
     * Recommended php extensions
     * 
     * @var array
     */
    protected $_recommendedPhpExtensions = array(
        //'apc' => 'The Alternative PHP Cache (APC) is a free and open caching engine for PHP. Use it for better performance',
        'zlib' => 'Used for data compression. Is required for normal functioning of Backups module',
        'zip' => 'Required to use compression of your backups with "Use Compression" enabled!',
    );
    
    protected $_encodedFiles = array(
    );

    const VALIDATION_SUCCESS = 'accept';
    const VALIDATION_WARNING = 'warning';
    const VALIDATION_ERROR   = 'del';

    const PHP_MIN_VERSION = '5.3';
    
    const MYSQL_TABLE_PREFIX = 'ur_';
    const MYSQL_MIN_VERSION  = '5.0';
    
    const TEMPLATE        = 'templates/index.php';
    const TEMPLATE_CONFIG = 'templates/application.inc';
    
    const BASE_URL = '/installation'; // without end slash
    
    const COLUMN_RIGHT = 'r';
    const COLUMN_LEFT  = 'l';
    
    /**
     * Initialization...
     * 
     * @param string $config 
     */
    public function __construct ($config, $installConfig)
    {        
        session_start();

        // Defaults
        $this->_config = $config;
        $this->_installConfig = $installConfig;
        $this->_initEvents();
        
        $this->validateSystemRequirements();
        $this->vaidateRequiredPhpExtensions();
        $this->vaidateRecommendedPhpExtensions();
        $this->validateServer();
        $this->validateWritableFolders();
    }
    
    /**
     * Magic wrapper :)
     * 
     * @return string
     */
    public function __toString ()
    {
        return $this->render();
    }
    
    /**
     * Render installation
     * 
     * @return string
     */
    public function render ()
    {
        ob_start();
        require_once realpath(dirname(__FILE__)) . '/' . self::TEMPLATE;
        return ob_get_clean();
    }
    
    /**
     * Make address with base URL
     * 
     * @param string $uri
     * @return string
     */
    public function baseUrl ($uri = '')
    {
        return $this->getBaseUrl() . self::BASE_URL . '/' . $uri;
    }
    
    /**
     * Validate required php extensions
     * 
     * @return void
     */
    public function vaidateRequiredPhpExtensions ()
    {
        foreach ($this->_requiredPhpExtensions as $extension => $message) {
            if (extension_loaded($extension)) {
                if ('pcre' == $extension) {
                    if (version_compare(PCRE_VERSION, '8') == -1) {
                        $this->addResult($extension, self::VALIDATION_ERROR, 'Required for core library to correctly compile some regular expressions (bb formatting codes). Requires PCRE 8.x');
                    } else {
                        $this->addResult($extension, self::VALIDATION_SUCCESS);
                    }
                } else {
                    $this->addResult($extension, self::VALIDATION_SUCCESS);
                }
            } else {
                $this->addResult($extension, self::VALIDATION_ERROR, $message);
            }
        }
    }
    
    /**
     * Validate recommended php extensions
     * 
     * @return void
     */
    public function vaidateRecommendedPhpExtensions ()
    {
        foreach ($this->_recommendedPhpExtensions as $extension => $message) {
            if (extension_loaded($extension)) {
                $this->addResult($extension, self::VALIDATION_SUCCESS);
            } else {
                $this->addResult($extension, self::VALIDATION_WARNING, $message);
            }
        }
    }
    
    /**
     * Validate apache(mod_rewrite)
     * 
     * @return void
    */
    public function validateServer ()
    {
        if (isset($_SERVER['HTTP_MOD_REWRITE'])) {
            if ($_SERVER['HTTP_MOD_REWRITE'] == 'On') {
                $this->addResult('Apache:mod_rewrite', self::VALIDATION_SUCCESS);
            } else {
                $this->addResult('Apache:mod_rewrite', self::VALIDATION_ERROR, 'We require mod_rewrite to be installed and enabled at Apache webserver for the links to be SEO-friendly');
            }
        }
    }
    
    /**
     * Validate writable folders
     * 
     * @return void
     */
    public function validateWritableFolders ()
    {
        foreach ($this->_writableFolders as $folder) {
            $realPath = ROOT_PATH . '/' . $folder;
            if (is_writable($realPath)) {
                if (is_dir($realPath)) {
                    $tmpFile = $realPath . '/' . uniqid(mt_rand()) . '.tmp';
                    if (!($f = @fopen($tmpFile, 'w+'))) {
                        $this->addResult('Writable ' . $folder, self::VALIDATION_ERROR, null);
                    } else {
                        fclose($f);
                        unlink($tmpFile);
                        $this->addResult('Writable ' . $folder, self::VALIDATION_SUCCESS, null);
                    }
                } else {
                    $this->addResult('Writable ' . $folder, self::VALIDATION_SUCCESS, null);
                }
            } else {
                $this->addResult('Writable ' . $folder, self::VALIDATION_ERROR, null);
            }
        }
    }
    
    /**
     * Add result
     * 
     * @param string $label
     * @param string $status
     * @param string $message
     * @return void
     */
    public function addResult ($label, $status, $message = null, $alt = null)
    {
        $result = (object) array(
            'status'  => $status,
            'message' => $message,
            'label'   => $label,
            'alt'     => $alt,
        );
        
        if ($status == self::VALIDATION_ERROR) {
            $this->_errors++;
        } elseif ($status == self::VALIDATION_WARNING) {
            $this->_warnings++;
        }
        
        $this->_results[] = $result;
    }
    
    /**
     * Has validation errors?
     * 
     * @var bool
     */
    public function hasErrors ()
    {
        return (bool) $this->_errors;
    }
    
    /**
     * Has validation warnings?
     * 
     * @var bool
     */
    public function hasWarnings ()
    {
        return (bool) $this->_warnings;
    }
    
    /**
     * Execute installation process
     * 
     * @return string
     */
    public function execute ()
    {
        if (isset($_POST['validate_db'])) {
            return (int) $this->validateDatabase();
        }
        
        if (isset($_POST['step'])) {
            return $this->install($_POST['step']);
        }
        
        if (array_key_exists('complete', $_REQUEST)) {
            return $this->renderComplete();
        }
        
        $this->validateEverything();
        $_SESSION = $_POST;
        return $this->render();
    }
    
    /**
     * Validate db connection
     * 
     * @return string
     */
    public function validateDb()
    {        
        $connection = Zend_Db::factory('Pdo_Mysql', array(
            'host' => $_SESSION['db_host'],
            'username' => $_SESSION['db_user'],
            'password' => $_SESSION['db_pass'],
            'dbname' => $_SESSION['db_name'],
            'autoQuoteIdentifiers' => false,
            'charset' => 'utf8',
        ));
        
    }
    
    /**
     * Installation process
     * 
     * @param string
     * @return string
     */
    public function install ($step)
    {
        $this->_getConnection();
        
        $config = new Zend_Config(array(
            'resources' => array('db' => array('params' => array('prefix' => $_SESSION['db_pref']))),
        ), true);
        $config = $config->merge(Singular_Runtime::extract("config"));
        Singular_Runtime::store("config", $config);
        Zend_Registry::set('config', $config);
        
        if (method_exists($this, $step)) {
            try {
                return call_user_func(array($this, $step));
            } catch (Exception $e) {
                if (!is_array($_SESSION['errors'])) {
                    $_SESSION['errors'] = array();
                }
                $_SESSION['errors'][] = $e->getMessage();
                return self::VALIDATION_ERROR;
            }
        }
        return self::VALIDATION_ERROR;
    }
    
    /**
     * Make Base URL
     * 
     * @return string
     */
    public function getBaseUrl ()
    {
        $port = $_SERVER['SERVER_PORT'] ? $_SERVER['SERVER_PORT'] : 80;
        $isSecure = 'ON' == strtoupper(@$_SERVER['HTTPS']);
        $protocol = ($isSecure || 443 == $port) ? 'https://' : 'http://';
        
        if (80 != $port && 443 != $port) {
            return rtrim($protocol . dirname($_SERVER['HTTP_HOST'] . ':' . $port . $_SERVER['SCRIPT_NAME']), '/');
        } else {
            return rtrim($protocol . dirname($_SERVER['HTTP_HOST'] . $_SERVER['SCRIPT_NAME']), '/');
        }
    }
    
    public function isSubFolder ()
    {
        $baseUrl = trim(dirname($_SERVER['HTTP_HOST'] . $_SERVER['SCRIPT_NAME']), '/');
        return preg_match('/\//', $baseUrl);
    }
    
    /**
     * Install default config
     * 
     * @return void
     */
    public function installConfig ()
    {
        $tpl = file_get_contents(realpath(dirname(__FILE__)) . '/' . self::TEMPLATE_CONFIG);
        $configData = str_replace(
            array('{db_host}', '{db_name}', '{db_user}', '{db_pass}', '{db_pref}', '{base_url}'),
            array($_SESSION['db_host'], $_SESSION['db_name'], $_SESSION['db_user'], $_SESSION['db_pass'], $_SESSION['db_pref'], $this->getBaseUrl()),
            $tpl
        );
        $config = fopen($this->_config . '.tmp', 'w+');
        fwrite($config, $configData);
        fclose($config);
        return self::VALIDATION_SUCCESS;
    }
    
    /**
     * Install firts administrator
     * 
     * @return void
     */
    public function installAdministrator ()
    {
        $db = $this->_getConnection();
        $table = $_SESSION['db_pref'] . 'users';
        
        $sql = "
            CREATE TABLE IF NOT EXISTS `{$table}` (
              `id` int(10) unsigned NOT NULL AUTO_INCREMENT,
              `full_name` varchar(100) NOT NULL,
              `password` varchar(32) DEFAULT NULL,
              `email` varchar(255) NOT NULL,
              `gravatar_email` varchar(255) DEFAULT NULL,
              `photo` varchar(255) DEFAULT NULL,
              `role` varchar(50) NOT NULL DEFAULT 'user',
              `is_active` tinyint(1) unsigned NOT NULL DEFAULT '1',
              `created_at` datetime NOT NULL,
              `updated_at` datetime DEFAULT NULL,
              `logged_at` datetime DEFAULT NULL,
              `hidden` tinyint(1) NOT NULL DEFAULT '0',
              `api_key` CHAR(32) NULL DEFAULT NULL,
              PRIMARY KEY (`id`),
              UNIQUE KEY `email` (`email`)
            ) ENGINE=MyISAM DEFAULT CHARSET=utf8 AUTO_INCREMENT=1 ;
        ";
        $db->exec($sql);
        
        $db->exec("DELETE FROM `$table` WHERE id = 1 LIMIT 1;");
        $db->insert($table, array(
            'id' => '1',
            'full_name' => $_SESSION['admin_email'],
            'password' => $this->_passwordHash($_SESSION['admin_password']),
            'email' => $_SESSION['admin_email'],
            'role' => 'administrator',
            'is_active' => '1',
            'created_at' => date('Y-m-d H:i:s'),
            'api_key' => System_Service_Api::generate(),
        ));
        
        $db->insert($table, array(
            'id' => '2',
            'full_name' => 'Anonymous',
            'password' => $this->_passwordHash($_SESSION['admin_password']),
            'email' => 'anonymous@localhost',
            'role' => 'user',
            'is_active' => '1',
            'created_at' => date('Y-m-d H:i:s'),
            'hidden'    => '1'
        ));

        return self::VALIDATION_SUCCESS;
    }
    
    /**
     * Install license
     * 
     * @return string
     */
    public function installLicense ()
    {
        return self::VALIDATION_SUCCESS;
    }
    
    /**
     * Install verify
     * 
     * @return string
     */
    public function installVerify ()
    {
        return self::VALIDATION_SUCCESS;
    }
    
    /**
     * Install ACL
     * 
     * @return string
     */
    public function installAcl ()
    {
        return self::VALIDATION_SUCCESS;
    }
    
    /**
     * Install default settings
     * 
     * @return string
     */
    public function installSettings ()
    {
        $db = $this->_getConnection();
        $table = $_SESSION['db_pref'] . 'settings';
        
        $sql = "
            CREATE TABLE IF NOT EXISTS `{$table}` (
              `id` int(10) unsigned NOT NULL AUTO_INCREMENT,
              `key` varchar(32) NOT NULL,
              `value` text,
              `type` varchar(50) NOT NULL DEFAULT 'text',
              `title` varchar(255) NOT NULL,
              `description` varchar(1000) DEFAULT NULL,
              `module` varchar(32) NOT NULL DEFAULT 'main',
              `section` varchar(32) NOT NULL DEFAULT 'main',
              PRIMARY KEY (`id`),
              UNIQUE KEY `key` (`key`)
            ) ENGINE=MyISAM  DEFAULT CHARSET=utf8 AUTO_INCREMENT=44 ;
        ";
        $db->exec($sql);
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'system_version',
            'value'       => '2.0',
            'type'        => 'text',
            'title'       => 'System Version',
            'description' => '',
            'module'      => 'system',
            'section'     => 'main'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'site_title',
            'value'       => $_SESSION['company_name'],
            'type'        => 'text',
            'title'       => 'Community Title',
            'description' => 'How will you call your community?',
            'module'      => 'system',
            'section'     => 'general'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'site_slogan',
            'value'       => '',
            'type'        => 'text',
            'title'       => 'Slogan',
            'description' => 'Give it the best shot!',
            'module'      => 'system',
            'section'     => 'general'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'site_description',
            'value'       => '',
            'type'        => 'textarea',
            'title'       => 'Description',
            'description' => 'Provide short description displayed at the top of the community. Leave blank if not required.',
            'module'      => 'system',
            'section'     => 'general'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'use_negative_votes',
            'value'       => '0',
            'type'        => 'yes_no',
            'title'       => 'Use Negative Votes',
            'description' => 'Use negative votes and comments on ideas and problems, so people can disagree with responses',
            'module'      => 'system',
            'section'     => 'responses'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'site_logo',
            'value'       => '',
            'type'        => 'image',
            'title'       => 'Community Logo',
            'description' => 'The image must be a GIF, JPEG or PNG that is smaller than 2 MB.',
            'module'      => 'system',
            'section'     => 'general'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'site_title_separator',
            'value'       => '|',
            'type'        => 'text',
            'title'       => 'Title Separator',
            'description' => 'Title Separator',
            'module'      => 'system',
            'section'     => 'main'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'site_default_timezone',
            'value'       => 'GMT',
            'type'        => 'text',
            'title'       => 'Default Timezone',
            'description' => 'Default Timezone',
            'module'      => 'system',
            'section'     => 'main'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'site_default_theme',
            'value'       => 'default',
            'type'        => 'themes',
            'title'       => 'Choose Theme',
            'description' => 'Select theme that will be used at your environment. You can customize it in public/themes',
            'module'      => 'system',
            'section'     => 'interface'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'site_default_admin_theme',
            'value'       => 'default',
            'type'        => 'admin_themes',
            'title'       => 'Default Admin Skin',
            'description' => 'Default Admin Skin',
            'module'      => 'system',
            'section'     => 'main'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'site_default_language',
            'value'       => 'en',
            'type'        => 'text',
            'title'       => 'Default Language',
            'description' => 'Default Language',
            'module'      => 'system',
            'section'     => 'main'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'site_default_locale',
            'value'       => 'en_US',
            'type'        => 'text',
            'title'       => 'Default Locale',
            'description' => 'Default Locale',
            'module'      => 'system',
            'section'     => 'main'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'paginator_page_range',
            'value'       => '10',
            'type'        => 'text',
            'title'       => 'Default paginator page range',
            'description' => 'Default paginator page range',
            'module'      => 'system',
            'section'     => 'main'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'paginator_item_count_per_page',
            'value'       => '10',
            'type'        => 'text',
            'title'       => 'Default paginator item count per page',
            'description' => 'Default paginator item count per page',
            'module'      => 'system',
            'section'     => 'main'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'paginator_scrolling_style',
            'value'       => 'Sliding',
            'type'        => 'text',
            'title'       => 'Default paginator scrolling style',
            'description' => 'Default paginator scrolling style',
            'module'      => 'system',
            'section'     => 'main'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'anonymous_votes',
            'value'       => '0',
            'type'        => 'yes_no',
            'title'       => 'Anonymous Votes',
            'description' => 'Accept anonymous votes if you don’t want users to register in the system.',
            'module'      => 'system',
            'section'     => 'general'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'email_connection_type',
            'value'       => 'sendmail',
            'type'        => 'select',
            'title'       => 'Connection Type',
            'description' => 'Native method uses your PHP to send emails. You can also use SMTP of your current or other server.',
            'module'      => 'system',
            'section'     => 'mailing'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'email_smtp_host',
            'value'       => '',
            'type'        => 'text',
            'title'       => 'SMTP Host',
            'description' => '',
            'module'      => 'system',
            'section'     => 'mailing'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'email_smtp_port',
            'value'       => '',
            'type'        => 'text',
            'title'       => 'Port',
            'description' => '',
            'module'      => 'system',
            'section'     => 'mailing'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'email_smtp_authentication',
            'value'       => '1',
            'type'        => 'yes_no',
            'title'       => 'Authentication',
            'description' => '',
            'module'      => 'system',
            'section'     => 'mailing'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'email_smtp_username',
            'value'       => '',
            'type'        => 'text',
            'title'       => 'Username',
            'description' => '',
            'module'      => 'system',
            'section'     => 'mailing'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'email_smtp_password',
            'value'       => '',
            'type'        => 'password',
            'title'       => 'Password',
            'description' => '',
            'module'      => 'system',
            'section'     => 'mailing'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'send_email_from_email',
            'value'       => $_SESSION['admin_email'],
            'type'        => 'email',
            'title'       => 'From Email',
            'description' => '',
            'module'      => 'system',
            'section'     => 'mailing'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'send_email_from_name',
            'value'       => $_SESSION['company_name'],
            'type'        => 'text',
            'title'       => 'From Name',
            'description' => '',
            'module'      => 'system',
            'section'     => 'mailing'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'email_smtp_security',
            'value'       => '',
            'type'        => 'select',
            'title'       => 'Security',
            'description' => '',
            'module'      => 'system',
            'section'     => 'mailing'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'email_is_bulk',
            'value'       => '1',
            'type'        => 'yes_no',
            'title'       => 'Bulk Messages',
            'description' => 'Send Notifications as bulk messages to not get autoresponds',
            'module'      => 'system',
            'section'     => 'mailing'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'send_notifications',
            'value'       => 'immediately',
            'type'        => 'radio',
            'title'       => 'Send Notifications',
            'description' => 'All Notifications could be sent once a day or immediately',
            'module'      => 'system',
            'section'     => 'user'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'photo_type',
            'value'       => 'avatar',
            'type'        => 'radio',
            'title'       => 'Avatar',
            'description' => '',
            'module'      => 'system',
            'section'     => 'user'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'notifications_created',
            'value'       => '1',
            'type'        => 'checkbox',
            'title'       => 'I\'ve Created',
            'description' => 'Notifications',
            'module'      => 'system',
            'section'     => 'user'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'notifications_posted_comments',
            'value'       => '1',
            'type'        => 'checkbox',
            'title'       => 'Posted Comments',
            'description' => 'Notifications',
            'module'      => 'system',
            'section'     => 'user'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'widget_text',
            'value'       => 'Feedback',
            'type'        => 'text',
            'title'       => 'Widget Text',
            'description' => '',
            'module'      => 'system',
            'section'     => 'widget'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'widget_design',
            'value'       => 'simple',
            'type'        => 'radio',
            'title'       => 'Widget Design',
            'description' => '',
            'module'      => 'system',
            'section'     => 'widget'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'widget_side',
            'value'       => 'left',
            'type'        => 'radio',
            'title'       => 'Widget Side',
            'description' => 'Choose widget location to be displayed',
            'module'      => 'system',
            'section'     => 'widget'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'widget_background_color',
            'value'       => '4F2D92',
            'type'        => 'color',
            'title'       => 'Background Color',
            'description' => '',
            'module'      => 'system',
            'section'     => 'widget'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'widget_border_color',
            'value'       => 'FFF830',
            'type'        => 'color',
            'title'       => 'Border Color',
            'description' => '',
            'module'      => 'system',
            'section'     => 'widget'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'widget_round_corners',
            'value'       => '1',
            'type'        => 'checkbox',
            'title'       => 'Round Corners',
            'description' => '',
            'module'      => 'system',
            'section'     => 'widget'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'widget_drop_shadow',
            'value'       => '1',
            'type'        => 'checkbox',
            'title'       => 'Drop Shadow',
            'description' => '',
            'module'      => 'system',
            'section'     => 'widget'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'widget_default_tab',
            'value'       => 'all_responses',
            'type'        => 'select',
            'title'       => 'Default Tab',
            'description' => 'What should be shown when opened',
            'module'      => 'system',
            'section'     => 'widget'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'widget_open',
            'value'       => 'widget',
            'type'        => 'radio',
            'title'       => 'Open',
            'description' => 'By clicking on tab user will go to',
            'module'      => 'system',
            'section'     => 'widget'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'send_notifications_once_time',
            'value'       => '12:00',
            'type'        => 'time',
            'title'       => 'Time',
            'description' => '',
            'module'      => 'system',
            'section'     => 'user'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'show_comments',
            'value'       => '1',
            'type'        => 'yes_no',
            'title'       => 'Comments Order',
            'description' => 'Show default sorting of comments in the system by dates',
            'module'      => 'system',
            'section'     => 'responses'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'force_to_comment_negative_vote',
            'value'       => '0',
            'type'        => 'yes_no',
            'title'       => 'Force to comment with negative vote',
            'description' => 'Each time user disagrees with response, he will be forwarded to add comment.',
            'module'      => 'system',
            'section'     => 'responses'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'last_frequently_activity',
            'value'       => '0',
            'type'        => 'int',
            'title'       => 'Last frequently activity',
            'description' => '',
            'module'      => 'system',
            'section'     => 'scheduler'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'last_hourly_activity',
            'value'       => '0',
            'type'        => 'int',
            'title'       => 'Last hourly activity',
            'description' => '',
            'module'      => 'system',
            'section'     => 'scheduler'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'last_daily_activity',
            'value'       => '0',
            'type'        => 'int',
            'title'       => 'Last daily activity',
            'description' => '',
            'module'      => 'system',
            'section'     => 'scheduler'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'ftp_host',
            'value'       => 'localhost',
            'type'        => 'text',
            'title'       => 'FTP Host',
            'description' => '',
            'module'      => 'system',
            'section'     => 'ftp'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'ftp_port',
            'value'       => '21',
            'type'        => 'text',
            'title'       => 'FTP Port',
            'description' => '',
            'module'      => 'system',
            'section'     => 'ftp'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'ftp_username',
            'value'       => '',
            'type'        => 'text',
            'title'       => 'FTP Username',
            'description' => '',
            'module'      => 'system',
            'section'     => 'ftp'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'ftp_password',
            'value'       => '',
            'type'        => 'password',
            'title'       => 'FTP Password',
            'description' => '',
            'module'      => 'system',
            'section'     => 'ftp'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'ftp_folder',
            'value'       => '',
            'type'        => 'text',
            'title'       => 'Useresponse Folder',
            'description' => '',
            'module'      => 'system',
            'section'     => 'ftp'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'top_responses',
            'value'       => 'popular',
            'type'        => 'select',
            'title'       => 'Top Respones',
            'description' => 'Select responses to be displayed in top on home page',
            'module'      => 'system',
            'section'     => 'responses'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'api_limit_rows',
            'value'       => '1000',
            'type'        => 'number',
            'title'       => 'Limit',
            'description' => 'Limit',
            'module'      => 'system',
            'section'     => 'api'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'api_is_enabled',
            'value'       => '1',
            'type'        => 'yes_no',
            'title'       => 'Enabled',
            'description' => 'Enabled',
            'module'      => 'system',
            'section'     => 'api'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'anonymous_responses',
            'value'       => '0',
            'type'        => 'yes_no',
            'title'       => 'Anonymous Responses',
            'description' => 'Allow users to add responses anonymously without authentication.',
            'module'      => 'system',
            'section'     => 'general'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'anonymous_captcha',
            'value'       => '0',
            'type'        => 'checkbox',
            'title'       => 'Use Captcha',
            'description' => '',
            'module'      => 'system',
            'section'     => 'general'
        ));
        
        Singular_Core::_("settings")->addSettings(array(
            'key'         => 'mobile_interface',
            'value'       => '0',
            'type'        => 'yes_no',
            'title'       => 'Use Mobile Interface',
            'description' => 'Advise optional interface to choose if using community from mobile',
            'module'      => 'system',
            'section'     => 'interface'
        ));
        
        return self::VALIDATION_SUCCESS;
    }
    
    /**
     * Install database
     * 
     * @return string
     */
    public function installDatabase ()
    {
        return self::VALIDATION_SUCCESS;
    }
    
    /**
     * Install modules
     * 
     * @return string
     */
    public function installModules ()
    {
        $db = $this->_getConnection();
        $table = $_SESSION['db_pref'] . 'modules';
        
        $sql = "
            CREATE TABLE IF NOT EXISTS `{$table}` (
              `id` int(3) unsigned NOT NULL AUTO_INCREMENT,
              `name` varchar(32) NOT NULL,
              `load_order` int(10) unsigned DEFAULT '0',
              `active` int(1) unsigned NOT NULL DEFAULT '1',
              `is_system` tinyint(1) NOT NULL DEFAULT '0',
              PRIMARY KEY (`id`),
              UNIQUE KEY `name` (`name`),
              FULLTEXT KEY `fulltext_name` (`name`)
            ) ENGINE=MyISAM  DEFAULT CHARSET=utf8 AUTO_INCREMENT=1 ;
        ";
        $db->exec($sql);
        
        $modulesDb = new System_Model_DbTable_Modules(array('name' => $table));
        foreach ($this->_modules as $order => $moduleName) {
            $moduleClass = Singular_Stdlib_Module::formatName($moduleName) . '_Module';
            $moduleFilePath = APPLICATION_PATH.DS.'modules'.DS.$moduleName.DS.Singular_Module::MODULE_FILE;
            require_once $moduleFilePath;
            $moduleInstance = new $moduleClass($moduleName);
            if ($moduleName != 'system') {
                $module = $modulesDb->createRow(array(
                    'name'       => $moduleName,
                    'active'     => 1,
                    'is_system'  => (int) $moduleInstance->isSystemModule(),
                    'load_order' => $order
                ));
                $module->save();
            }
            $moduleInstance->install();
        }
        return self::VALIDATION_SUCCESS;
    }
    
    /**
     * Install environment
     * 
     * @return string
     */
    public function installEnv ()
    {
        if (!empty($this->_encodedFiles)) {
            foreach ($this->_encodedFiles as $file) {
                if (version_compare(PHP_VERSION, '5.3') >= 0) {
                    $unsetFile = ROOT_PATH.'/'.$file.'52';
                    unlink($unsetFile);
                } else {
                    $unsetFile = ROOT_PATH.'/'.$file;
                    unlink($unsetFile);
                    rename(ROOT_PATH.'/'.$file.'52', ROOT_PATH.'/'.$file);
                }
            }
        }
        return self::VALIDATION_SUCCESS;
    }
    
    /**
     * Final result
     * 
     * @return string
     */
    public function getFinalResult ()
    {
        if ($this->hasErrors()) {
            return '<div class="text-area color2"><p>There are some problems with environment! Please resolve issues to continue.</p></div>';
        } elseif ($this->hasWarnings()) {
            return '<div class="text-area color1"><p>UseResponse can be installed, but there are some recommendations.</p></div>';
        } else {
            return '<div class="text-area color1"><p>All checks are passed. UseResponse can be installed on this server!</p></div>';
        }
    }
    
    /**
     * Render item of message
     * 
     * @param string $message
     * @return string 
     */
    public function renderMessage ($message)
    {
        return $message ? '<span class="ico-small hint" title="'.$message.'"></span>' : '';
    }
    
    /**
     * Render item of result
     * 
     * @param object $result
     * @param string $message
     * @param string $color
     * @return string 
     */
    public function renderResult ($result, $message, $color)
    {
        $alt = '';
        $questionClass = '';
        if (isset($result->alt) && $result->alt) {
            $alt = '<span title="'.$result->alt.'">'.$result->label.'</span>';
            $questionClass = 'question';
        } else {
            $alt = $result->label;
        }
        
        return '<div class="'.$questionClass.' obj'.$color.'">'.$alt.$message.'<span class="ico-small '.$result->status.'"></span></div>';
    }
    
    /**
     * Render all results
     * 
     * @param string $column
     * @return string 
     */
    public function renderResults ($column)
    {
        $results = array();
        $center = ceil(count($this->_results) / 2);
        
        if ($column == self::COLUMN_LEFT) {
            $results = array_slice($this->_results, 0, $center);
        } else {
            $results = array_slice($this->_results, $center);
        }

        $out = array();
        foreach ($results as $key => $result) {
            $color = $key & 1 ? ' color-obj' : '';
            $message = $this->renderMessage($result->message);
            $out[] = $this->renderResult($result, $message, $color);
        }
        
        return implode(PHP_EOL, $out);
    }
    
    /**
     * Render installation result
     * 
     * @return string
     */
    public function renderComplete ()
    {
        rename($this->_config . '.tmp', $this->_config);
        chmod($this->_config, 0666);
        
        if (isset($_SESSION['errors']) && is_array($_SESSION['errors'])) {
            $msg  = '<div class="text-area color2">';
            $msg .= '  <p>An error occurred during installation, try again.</p>';
            $msg .= '</div>';
            foreach ($_SESSION['errors'] as $message) {
                $msg .= '<p>' . $message . '</p>';
            }
            unset($_SESSION['errors']);
            return $msg;
        } else {
            return '
                <div class="text-area color1">
                    <p>Congratulation! Now you can use your <a href="'.$this->getBaseUrl().'" class="feedback" target="_blank">Customer Feedback Platform!</a></p>
                </div>
                <p>Make sure to delete installation folder. If you experience any problems please <a href="mailto:support@useresponse.com">contact our support</a> or use <a href="http://www.community.useresponse.com">support platform!</a></p>
            ';
        }
    }

    /**
     * Validate system requirements
     * 
     * @return void
     */
    public function validateSystemRequirements ()
    {
        // Validate PHP version
        $phpMinVer = self::PHP_MIN_VERSION;
        $phpVer = PHP_VERSION;
        if (version_compare($phpVer, $phpMinVer) == -1) {
            $this->addResult("PHP $phpVer", self::VALIDATION_ERROR, "Minimal PHP version required in order to run UseResponse is PHP $phpMinVer");
        } else {
            $this->addResult("PHP $phpVer", self::VALIDATION_SUCCESS);
        }
        
        // Validate MySQL
        if (function_exists('mysql_get_client_info')) {
            $mysqlMinVer = self::MYSQL_MIN_VERSION;
            $mysqlVer = mysql_get_client_info();
            preg_match("@[\.0-9]+@", $mysqlVer, $matches);
            if (version_compare($matches[0], $mysqlMinVer) >= 0) {
                $this->addResult("MySQL $mysqlVer", self::VALIDATION_SUCCESS);
            } else {
                $this->addResult("MySQL $mysqlVer", self::VALIDATION_ERROR, "Please update your MYSQL. Minimum required version is $mysqlMinVer!");
            }
        } else {
            $this->addResult("MySQL", self::VALIDATION_ERROR, "UseResponse uses MYSQL as primary data storage. You won't be able to install UseResponse without MYSQL");
        }
        
        // Validate ZendGuardLoader/Zend Optimizer
        if (version_compare(PHP_VERSION, '5.3') >= 0) {
            if (!extension_loaded('Zend Guard Loader')) {
                $this->addResult('Zend Guard Loader / Zend Optimizer', self::VALIDATION_ERROR, 'Zend Guard is required in order to run UseResponse and use all system advantages (licensing, removing copyrights)');
            } else {
                $this->addResult('Zend Guard Loader / Zend Optimizer', self::VALIDATION_SUCCESS);
            }
        } else {
            if (!extension_loaded('Zend Optimizer')) {
                $this->addResult('Zend Guard Loader / Zend Optimizer', self::VALIDATION_ERROR, 'Zend Optimizer is required in order to run UseResponse and use all system advantages (licensing, removing copyrights');
            } else {
                $this->addResult('Zend Guard Loader / Zend Optimizer', self::VALIDATION_SUCCESS);
            }
        }
        
        // Validate safe mode
        if (version_compare(PHP_VERSION, '5.3.0') == -1) {
            if (ini_get('safe_mode')) {
                $this->addResult('PHP safe mode off', self::VALIDATION_ERROR, 'PHP configuration option to apply securuty limitations for script\'s execution environment');
            } else {
                $this->addResult('PHP safe mode off', self::VALIDATION_SUCCESS);
            }
        } else {
            $this->addResult('PHP safe mode off', self::VALIDATION_SUCCESS);
        }
    }
    
    /**
     * Validate everything
     * 
     * @return bool
     */
    public function validateEverything ()
    {
        $this->_isExecute = true;
        if (!$this->validateDatabase()) {
            $this->_everythingError = 'Can\'t connect to database with provided access!';
        } elseif (!$this->validateAgreements()) {
            $this->_everythingError = 'You need to Accept License Agreement to continue!';
        }
        $this->_isValid = $this->_everythingError ? false : true;
        return $this->_isValid;
    }
    
    /**
     * Validate database
     * 
     * @return bool
     */
    public function validateDatabase ()
    {
        $connection = @mysql_connect($_POST['db_host'], $_POST['db_user'], $_POST['db_pass']);
        if ($connection) {          
            $mysqlVersion = @mysql_get_server_info($connection);
            if (preg_match("@[\.0-9]+@", $mysqlVersion, $matches)) {
                if (version_compare($matches[0], self::MYSQL_MIN_VERSION) >= 0) {
                    if (@mysql_select_db($_POST['db_name'], $connection)) {
                        return true;
                    }
                }
            }
        }
        return false;
    }
    
    /**
     * Get db connection
     * 
     * @return Zend_Db_Adapter_Pdo_Mysql
     */
    protected function _getConnection ()
    {
        if ($this->_connection instanceof Zend_Db) {
            return $this->_connection;
        }
        
        $this->_connection = Zend_Db::factory('Pdo_Mysql', array(
            'host' => $_SESSION['db_host'],
            'username' => $_SESSION['db_user'],
            'password' => $_SESSION['db_pass'],
            'dbname' => $_SESSION['db_name'],
            'autoQuoteIdentifiers' => false,
            'charset' => 'utf8',
        ));
        
        Singular_Db_Table_Abstract::setDefaultAdapter($this->_connection);
        
        return $this->_connection;
    }
    
    /**
     * Validate license
     * 
     * @return bool
     */
    public function validateLicense ()
    {
        return true;
    }
    
    /**
     * Validate agreements
     * 
     * @return bool
     */
    public function validateAgreements ()
    {
        return isset($_POST['agreement']) && $_POST['agreement'] ? true : false;
    }

    /**
     * Render item of field
     * 
     * @param string $type
     * @param string $name
     * @param string $defaulValue
     * @param string $validator
     * @param array $options
     * @return string 
     */
    public function renderField ($type, $name, $defaulValue = '', $validator = '', $options = null, $class = '')
    {
        if ($this->_isExecute) {
            switch ($validator) {
                case 'empty':
                    if (@empty($_POST[$name])) {
                        return '<input type="'.$type.'" name="'.$name.'" value="'.$_POST[$name].'" class="error" />';
                    } else {
                        return '<input type="'.$type.'" name="'.$name.'" value="'.$_POST[$name].'" />';
                    }
                break;
                case 'yes':
                    if (isset($_POST[$name]) && $_POST[$name]) {
                        $checked = $_POST[$name] ? 'checked="checked"' : '';
                        return '<label class="'.$options['class'].'"> '.$options['label'].' <input type="'.$type.'" name="'.$name.'" value="'.$_POST[$name].'" '.$checked.' /></label>';
                    } else {
                        return '<label class="'.$options['class'].' error"> '.$options['label'].' <input type="'.$type.'" name="'.$name.'" value="1" /></label>';
                    }
                break;
                case 'file':
                    if ($this->_isUploadedLicense) {
                        return '<input type="'.$type.'" name="'.$name.'" value="" class="inputFile" />';
                    } else {
                        return '<input type="'.$type.'" name="'.$name.'" value="" class="error inputFile" />';
                    }
                break;
                case 'email':
                    if (preg_match('#^[a-zA-Z0-9._-]+@[a-zA-Z0-9._-]+\.([a-zA-Z]{2,4})$#ui', $_POST[$name])) {
                        return '<input type="'.$type.'" name="'.$name.'" value="'.$_POST[$name].'" />';
                    } else {
                        return '<input type="'.$type.'" name="'.$name.'" value="'.$_POST[$name].'" class="error" />';
                    }
                break;
                case '':
                default:
                    return '<input type="'.$type.'" name="'.$name.'" value="'.$_POST[$name].'" />';
                break;
            }
        } else {
            if (!empty($options)) {
                return '<label class="'.$options['class'].'"> '.$options['label'].' <input type="'.$type.'" name="'.$name.'" value="'.$defaulValue.'" /></label>';
            } else {
                return '<input type="'.$type.'" name="'.$name.'" value="'.$defaulValue.'" class="'.$class.'" />';
            }
        }
    }
    
    /**
     * Render partial of template
     * 
     * @return string
     */
    public function partial ($name)
    {
        return include realpath(dirname(__FILE__)) . '/templates/' .$name;
    }
    
    /**
     * Password hash generator
     *
     * @param  string $password
     * @return string
     */
    protected function _passwordHash ($password)
    {
        $password = strtolower($password);
        return md5(
            str_repeat(
                md5($password) . strrev($password) . sha1($password),
                strlen($password)
            )
        );
    }
    
    /**
     * Initialization event logic
     * 
     * @return void
     */
    protected function _initEvents ()
    {
        foreach ($this->_modules as $module) {
            $handlersPath = MODULES_PATH . DS . $module . DS . 'handlers';
            $handlers = glob($handlersPath . DS . '*.php');
            foreach ($handlers as $handler) {
                $handlerClass = Singular_Stdlib_Module::formatName($module) . '_Handler_' . ucfirst(basename($handler, '.php'));
                if (!is_subclass_of($handlerClass, 'Singular_Event_Listener_Abstract')) continue;
                $handlerClass::bindListeners();
            }
        }
    }
}
