<?php
/**************************************************************
 * Copyright USWebStyle Inc., 2011-2012. All Rights Reserved.
 **************************************************************
 * NOTICE:  This source code is the property of USWebStyle, Inc.
 * In no case shall you rent, lease, lend, redistribute
 * any of below source code to a 3rd party individual or entity.
 *
 * @category   System
 * @package    System_Service
 * @copyright  USWebStyle Inc., 2011-2012 (http://www.uswebstyle.com)
 * @license    http:/www.useresponse.com/licensing (Commercial)
 */

/**
 * Upgrade service
 *
 * @category   System
 * @package    System_Service
 */
class System_Service_Upgrade
{
    const VERSION_URL = "http://www.useresponse.com/version";
    const ARCHIVE_URL = "http://www.useresponse.com/update?domain=%s&hash=%s";
    
    static public $ftpFolder = ROOT_PATH;
    
    /**
     * Returns array
     *
     * @static
     * @return mixed
     */
    static public function testFtpConnection($params = array())
    {
        if (!extension_loaded("Ftp")) {
            return array(false, "Auto-Updater requires PHP Ftp Extention!");
        }
        
        $client = self::getFtpClient($params);
        
        try {
            $client->getConnection();
        } catch (Singular_Ftp_Exception $e) {
            return array(false, "Can't connect to local server. Check FTP Settings!");
        }
        
        try {
            $client->getDirectory(self::$ftpFolder)->getContents();
        } catch (Singular_Ftp_Exception $e) {
            return array(false, "Installation directory not found!");
        }
        
        $testFileName = ROOT_PATH . DS . 'work' . DS . 'test';
        file_put_contents($testFileName, "");
        
        try {
            $applicationDir = $client->getDirectory(self::$ftpFolder . DS . 'application');
            $testFtpFile = $applicationDir->put($testFileName, FTP_BINARY);
        } catch (Singular_Ftp_Exception $e) {
            return array(false, "Not enough permissions to upgrade files!");
        }
        
        $testFtpFile->delete();
        unlink($testFileName);

        return array(true, "Connection works!");
    }
    
    static public function createBackup()
    {
        //root
        $backupDir = ROOT_PATH . DS . 'work' . DS . 'upgrade' . DS . 'backup';
        if (!is_dir($backupDir)) {
            mkdir($backupDir);
        }
        $archiveName = ROOT_PATH . DS . 'work' . DS . 'upgrade' . DS . 'backup.zip';
        if (file_exists($archiveName)) {
            unlink($archiveName);
        }
        $exceptions = array('application', 'work', 'public', 'installation');
        $items = new DirectoryIterator(ROOT_PATH);
        foreach ($items as $item) {
            if (!$item->isDot()) {
                if (!in_array($item->getBasename(), $exceptions)) {
                    if ($item->isFile()) {
                        copy($item->getRealPath(), $backupDir . DS . $item->getBasename());
                    } else {
                        System_Service_Files::copyDir($item->getRealPath(), $backupDir . DS . $item->getBasename());
                    }
                }
            }
        }
        //application
        $backupDir = ROOT_PATH . DS . 'work' . DS . 'upgrade' . DS . 'backup' . DS . 'application';
        mkdir($backupDir);
        $exceptions = array('temp');
        $items = new DirectoryIterator(APPLICATION_PATH);
        foreach ($items as $item) {
            if (!$item->isDot()) {
                if (!in_array($item->getBasename(), $exceptions)) {
                    if ($item->isFile()) {
                        copy($item->getRealPath(), $backupDir . DS . $item->getBasename());
                    } else {
                        System_Service_Files::copyDir($item->getRealPath(), $backupDir . DS . $item->getBasename());
                    }
                }
            }
        }
        //public
        $backupDir = ROOT_PATH . DS . 'work' . DS . 'upgrade' . DS . 'backup' . DS . 'public';
        mkdir($backupDir);
        $exceptions = array('static', 'files');
        $items = new DirectoryIterator(ROOT_PATH . DS . 'public');
        foreach ($items as $item) {
            if (!$item->isDot()) {
                if (!in_array($item->getBasename(), $exceptions)) {
                    if ($item->isFile()) {
                        copy($item->getRealPath(), $backupDir . DS . $item->getBasename());
                    } else {
                        System_Service_Files::copyDir($item->getRealPath(), $backupDir . DS . $item->getBasename());
                    }
                }
            }
        }
        
        return self::backupDb($backupDir . 'db.sql');
    }
    
    static public function getFtpClient($params = array())
    {
        if (empty($params)) {
            $settings = Singular_Core::_("settings")->getModuleSettings('system', 'ftp');
            foreach ($settings as $setting) {
                $params[$setting->key] = $setting->value;
            }
        }
        self::$ftpFolder = $params['ftp_folder'];
        $ftp = new Singular_Ftp(
                $params['ftp_host'],
                $params['ftp_username'],
                $params['ftp_password'],
                $params['ftp_port']
            );
        return $ftp;
    }

    static public function uploadNewVersion ()
    {
        $domain = $_SERVER['HTTP_HOST'];
        $hash = md5(trim(file_get_contents(APPLICATION_PATH . DS . 'configs' . DS . 'license.key')));

        $url = sprintf(System_Service_Upgrade::ARCHIVE_URL, $domain, $hash);
        $client = new Zend_Http_Client($url);
        $request = $client->request();
        $status = $request->getStatus();
        $response = $request->getBody();
        if (strlen($response) < 500) {
            die($response);
        }
        if ($status == 200) {
            $filename = ROOT_PATH . '/work/upgrade/for-upload.zip';
            file_put_contents($filename, $response);
            if (filesize($filename)) {
                return true;
            }
            @unlink($filename);
        }
        return false;
    }

    static public function extractNewVersion()
    {
        $filename = ROOT_PATH . '/work/upgrade/for-upload.zip';
        $zip = new ZipArchive();
        if ($zip->open($filename) === true) {
            $zip->extractTo(ROOT_PATH . '/work/upgrade/system/');
            @unlink($filename);
            //move localization
            $sourceDir = ROOT_PATH . '/work/upgrade/system/application/localization/';
            $destinationDir = ROOT_PATH . '/work/upgrade/localization/';
            System_Service_Files::copyDir($sourceDir, $destinationDir);
            System_Service_Files::removeDir($sourceDir);
            //move themes
            $sourceDir = ROOT_PATH . '/work/upgrade/system/public/themes/';
            $destinationDir = ROOT_PATH . '/work/upgrade/themes/';
            System_Service_Files::copyDir($sourceDir, $destinationDir);
            System_Service_Files::removeDir($sourceDir);
            //remove files
	    System_Service_Files::removeDir(ROOT_PATH . '/work/upgrade/system/installation/');
	    System_Service_Files::removeDir(ROOT_PATH . '/work/upgrade/system/public/files/');
            @unlink(ROOT_PATH . '/work/upgrade/system/.htaccess');
            @unlink(ROOT_PATH . '/work/upgrade/system/work/.htaccess');
            
            return true;
        }
        return false;
    }
    
    static public function upgradeSystem()
    {
        $source = ROOT_PATH . '/work/upgrade/system/';
        $client = self::getFtpClient();
        $dest = $client->getDirectory(self::$ftpFolder . '/');
        $step = 0;
        try {
            $log = new stdClass();
            $log->update = array();
            $log->create = array();
            $step++;
            self::copyFromLocalToFtpRecursive($source, $dest, $log);
            $step++;
            self::upgradeDbConfig();
            
        } catch (Zend_Exception $e) {
            try {
                self::restoreFiles($log);
                if ($step == 2) {
                    $filename = ROOT_PATH . DS . 'work' . DS . 'upgrade' . DS . 'backup' . DS . 'public' . DS . 'db.sql';
                    self::restoreDb($filename);
                }
            } catch (Zend_Exception $e) {
                return false;
            }
            return false;
        }

        Singular_Core::_('Cache')->cleanAll();

        return true;
    }

    static public function upgradeThemes()
    {
        $client = self::getFtpClient();
        $source = ROOT_PATH . '/work/upgrade/themes/';
        $dest = $client->getDirectory(self::$ftpFolder . '/public/themes/');
        return self::copyFromLocalToFtpRecursive($source, $dest);
    }
    
    static public function upgradeLocalization()
    {
        $client = self::getFtpClient();
        $source = ROOT_PATH . '/work/upgrade/localization/';
        $dest = $client->getDirectory(self::$ftpFolder . '/application/localization/');
        return self::copyFromLocalToFtpRecursive($source, $dest);
    }
    
    static public function completion()
    {
        $zip = new ZipArchive();
        $archiveName = ROOT_PATH . DS . 'work' . DS . 'upgrade' . DS . 'backup.zip';
        if ($zip->open($archiveName, ZIPARCHIVE::CREATE) === true) {
            $source = ROOT_PATH . DS . 'work' . DS . 'upgrade' . DS . 'backup';
            $dest = DS;
            self::addDirToZip($source, $dest, $zip);
            $zip->close();
        }
        
        System_Service_Files::removeDir(ROOT_PATH . DS . 'work' . DS . 'upgrade' . DS . 'backup');
        System_Service_Files::removeDir(ROOT_PATH . DS . 'work' . DS . 'upgrade' . DS . 'system');
        System_Service_Files::removeDir(ROOT_PATH . DS . 'work' . DS . 'upgrade' . DS . 'localization');
        System_Service_Files::removeDir(ROOT_PATH . DS . 'work' . DS . 'upgrade' . DS . 'themes');
        
        Singular_Core::_('Cache')->cleanAll();
        
        return true;
    }
    
    static public function restoreDb($filename)
    {
        $sql = file($filename);
        $adapter = Zend_Db_Table::getDefaultAdapter();

        if (!empty($sql)) {
            $query = "";
            foreach ($sql as $sqlLine) {
                $sqlLine = trim($sqlLine);
                if($sqlLine != "" && strpos($sqlLine, "--") !== 0) {
                    $query .= $sqlLine;
                    if (preg_match("/;$/", $sqlLine)) {
                        $result = $adapter->query($query);
                        $query = "";
                    }
                }
            }
            return true;
        } else {
            return false;
        }
    }

    static public function restoreFiles($log)
    {
        $backupDir = ROOT_PATH . DS . 'work' . DS . 'upgrade' . DS . 'backup';
        if (!is_dir($backupDir)) {
            return false;
        }
        $client = self::getFtpClient();
        $len = strlen(ROOT_PATH);
        
        if (!empty($log->update)) {
            foreach ($log->update as $item) {
                $subPath = substr($item, $len);
                $file = new Singular_Ftp_File($item, $client);
                $file->put($backupDir . DS .$subPath , FTP_BINARY);
            }
        }
        
        if (!empty($log->create)) {
            foreach ($log->create as $item) {
                $subPath = substr($item, $len);
                $file = new Singular_Ftp_File($item, $client);
                $file->delete();
            }
        }
    }
    
    /**
     * Create SQL dump from database tables.
     *
     * @param  string $filename   SQL dump filename
     * @return bool
     */
    static public function backupDb($filename)
    {
        $dbPrefix = Zend_Registry::get('config')->resources->db->params->prefix;
        $adapter = Zend_Db_Table::getDefaultAdapter();
        if ($dbPrefix) {
            $tables = $adapter->fetchCol("SHOW TABLES LIKE '$dbPrefix%'");
        } else {
            $tables = $adapter->fetchCol('SHOW TABLES');
        }
        $output = '';
        if (is_array($tables)) {
            foreach ($tables as $tableName) {
                // dump_structure
                $createTable = $adapter->fetchRow("SHOW CREATE TABLE $tableName");
                $output .= "DROP TABLE IF EXISTS `$tableName`;\n"
                        . $createTable['Create Table'] . ";\n\n";

                // dump_data
                $res = $adapter->query("SELECT * FROM `$tableName`");

                $insertedValues = '';
                while ($row = $res->fetch(Zend_Db::FETCH_NUM)) {
                    $values = '';
                    foreach ($row as $value) {
                        $values .= $values ? ',' : '';
                        $values .= is_null($value) ? "NULL" : $adapter->quote($value);
                    }
                    $insertedValues .= ($insertedValues) ? ',' : '';
                    $insertedValues .= '(' . $values . ')';
                    $output .= "INSERT INTO `$tableName` VALUES $insertedValues;\n";
                    $insertedValues = '';
                }
            }
        }        
        if (file_put_contents($filename, $output)) {
            return true;
        }
        return false;
    }


    /**
     * Add to zip archive from given folder.
     *
     * @param string $sourceDir
     * @param string $destinationDir
     */
    static public function addDirToZip ($sourceDir, $destinationDir, $zip)
    {
        if (is_dir($sourceDir)) {
            $zip->addEmptyDir($destinationDir);
            $items = new DirectoryIterator($sourceDir);
            foreach ($items as $item) {
                if (!$item->isDot()) {
                    if ($item->isFile()) {
                        $zip->addFile($item->getRealPath(), $destinationDir . '/' . $item->getBasename());
                    } else {
                        self::addDirToZip($item->getRealPath(), $destinationDir . '/' . $item->getBasename(), $zip);
                    }
                }
            }
        }
    }
    
    
    /**
     * Returns boolean
     *
     * @static
     * @return mixed
     */
    static protected function copyFromLocalToFtpRecursive($sourceDir, $destinationDir, $log)
    {
        if (is_dir($sourceDir)) {
            $items = new DirectoryIterator($sourceDir);
            $ftpItems = $destinationDir->getContents();
            foreach ($items as $item) {
                if (!$item->isDot()) {
                    
                    $newFtpItem = null;
                    foreach ($ftpItems as $ftpItem) {
                        if ($ftpItem->name == $item->getBasename()) {
                            $newFtpItem = $ftpItem;
                        }
                    }
                    
                    if ($item->isFile()) {
                        $file = $destinationDir->put($item->getRealPath(), FTP_BINARY, $item->getBasename());
                        
                        if ($newFtpItem === null) {
                            $log->create[$item->getRealPath()] = $file->path;
                            $file->chmod($item->getPerms());
                        } else {
                            $log->update[$item->getRealPath()] = $file->path;
                        }
                    } else {
                        if ($newFtpItem === null) {
                            $newFtpItem = $destinationDir->getDirectory($item->getBasename())->create($item->getPerms());
                        }
                        self::copyFromLocalToFtpRecursive($item->getRealPath(), $newFtpItem, $log);
                    }
                }
            }
        }
        return true;
    }
    
    static public function upgradeDbConfig()
    {
        Singular_Core::_('Cache')->cleanAll();
        $curVersion = Singular_Core::_("settings")->getSetting('system_version');
        if ($curVersion === null) {
            $curVersion = "1.0.2";
        } else {
            $curVersion = str_pad($curVersion, 5, '.0', STR_PAD_RIGHT);
        }
        $directory = ROOT_PATH . '/upgrade/instructions';
        $items = new DirectoryIterator($directory);
        foreach ($items as $item) {
            if (!$item->isDot()) {
                if ($item->isFile()) {
                    $basename = $item->getBasename();
                    if (preg_match("|^upgrade_([0-9]*)\.php$|i", $basename, $matches)) {
                        $namber = (int)$matches[1];
                        $version = $matches[1][0] . '.' . $matches[1][1] . '.' . $matches[1][2];
                        $instructions[$namber] = array(
                            'version' => $version,
                            'filename' => $basename
                        );
                    }
                }
            }
        }
        ksort($instructions);
        
        $configFile = APPLICATION_PATH . DS . 'configs' . DS . 'application.inc';
        $writer = new Zend_Config_Writer_Array();
        $configArray = include $configFile;
        $config = new Zend_Config($configArray, true);
        
        $tablePrefix = Singular_Runtime::extract("config")->resources->db
            ->params->prefix;
        $adapter = Zend_Db_Table::getDefaultAdapter();
        
        $installedModules = Singular_Module::getInstance()->getInstalledModules();

        $result = true;
        foreach ($instructions as $nubmer=>$data) {
            if (version_compare($curVersion, $data['version']) < 0) {
                $result = $result && include_once ROOT_PATH . '/upgrade/instructions/' . $data['filename'];
            }
        }
        
        Singular_Core::_('Cache')->cleanAll();
        
        $module = new System_Module('system');
        Singular_Core::_("settings")->changeSetting('system_version', $module->getVersion());
        
        return $result;
    }
}
